import hashlib

from django.db import models as django_models

from intranet.femida.src.vacancies import models


def create_publication_subscription(data):
    fields_to_set = (
        'abc_services',
        'professions',
        'skills',
        'cities',
    )
    fields_data = {}

    for field in fields_to_set:
        if field in data:
            fields_data[field] = data.pop(field)
    subscription = models.PublicationSubscription.objects.create(**data)
    for field, qs in fields_data.items():
        getattr(subscription, field).set(qs)


def _item_to_string(item):
    if isinstance(item, django_models.QuerySet):
        subitem_ids = sorted(i.id for i in item)
        return str(subitem_ids)
    if isinstance(item, django_models.Model):
        return str(item.id)
    return str(item)


def _subscription_to_string(subscription_fields):
    """
    Формирует строковое представление для фильтра по публикациям.
    :param subscription_fields: Словарь из полей, каждое из которых может быть
        QuerySet, Model или взаимнооднозначно отображаемым в строку объектом.
    :return: Строка вида "['department': 2, 'only_active': True, 'skills': [1, 2]]"
    """
    fields = [
        '{}: {}'.format(
            field_name,
            _item_to_string(subscription_fields[field_name]),
        )
        for field_name in sorted(subscription_fields)
        if subscription_fields[field_name]
    ]
    return str(fields)


def get_subscription_hash(subscription_fields):
    """
    Формирует sha1 хэш для фильтра по публикациям.
    Свойства хэша:
        * порядок полей в subscription_fields не важен
        * если поле приводится к False, это эквивалентно отсутствию поля
    :param subscription_fields: dict: Словарь из полей, каждое из которых может быть
        QuerySet, Model или взаимнооднозначно отображаемым в строку объектом.
    :return: str: Хэш sha1 по полям subscription_fields
    """
    subscription_string = _subscription_to_string(subscription_fields)
    return hashlib.sha1(subscription_string.encode('utf-8')).hexdigest()
