import logging
import waffle

from django.conf import settings
from django.template import loader
from django.utils.functional import cached_property

from intranet.femida.src.applications.controllers import close_vacancy_applications
from intranet.femida.src.core.workflow import Action, Workflow, WorkflowError
from intranet.femida.src.notifications.vacancies import (
    VacancyApprovedNotification,
    VacancyClosedNotification,
    VacancyResumedNotification,
    VacancySuspendedNotification,
    VacancyUpdatedNotification,
)
from intranet.femida.src.interviews.choices import APPLICATION_RESOLUTIONS
from intranet.femida.src.oebs.api import get_budget_position, BudgetPositionError
from intranet.femida.src.oebs.choices import BUDGET_POSITION_STATUSES
from intranet.femida.src.oebs.tasks import update_oebs_assignment_task
from intranet.femida.src.offers.choices import OFFER_STATUSES
from intranet.femida.src.publications.controllers import archive_vacancy_publications
from intranet.femida.src.staff.bp_registry import BPRegistryAPI, BPRegistryError
from intranet.femida.src.startrek.operations import (
    IssueCommentOperation,
    IssueTransitionOperation,
    IssueTransitionOrUpdateOperation,
    IssueUpdateOperation,
)
from intranet.femida.src.startrek.utils import (
    StartrekError,
    KnownStartrekError,
    TransitionFailed,
    TransitionDoesNotExist,
    IssueInvalidStatus,
    TransitionEnum,
    StatusEnum,
    get_issue,
)
from intranet.femida.src.vacancies.choices import (
    VACANCY_STATUSES,
    VACANCY_RESOLUTIONS,
    VACANCY_TYPES,
)
from intranet.femida.src.vacancies.controllers import update_or_create_vacancy
from intranet.femida.src.vacancies.models import Vacancy
from intranet.femida.src.vacancies.serializers import VacancyDiffEstimationSerializer
from intranet.femida.src.vacancies.bp_registry.serializers import (
    BPRegistryVacancyCancellationSerializer,
)
from intranet.femida.src.vacancies.signals import vacancy_approved


logger = logging.getLogger(__name__)


class VacancyActionBase(Action):

    def has_permission(self):
        return self.user.is_recruiter


class StartrekMixin:

    def is_startrek_operations_disabled(self):
        return (
            waffle.switch_is_active('ignore_job_issue_workflow')
            or not self.instance.startrek_key
        )

    def check_startrek_issue_status(self, expected_statuses):
        if waffle.switch_is_active('ignore_job_issue_workflow'):
            return True

        try:
            issue = get_issue(self.instance.startrek_key)
            return issue.status.key in expected_statuses
        except StartrekError as exc:
            raise WorkflowError(exc.message)

    def push_comment_to_ticket(self, comment):
        if self.is_startrek_operations_disabled():
            return

        operation = IssueCommentOperation(self.instance.startrek_key)
        try:
            operation(comment)
        except StartrekError:
            operation.delay(comment)

    def change_ticket_status(self, transition, **fields):
        if self.is_startrek_operations_disabled():
            return

        operation = IssueTransitionOperation(self.instance.startrek_key)
        fields['transition'] = transition

        error_message = None
        retry = True
        try:
            operation(**fields)
        except TransitionDoesNotExist as exc:
            error_message = exc.message
            # Если перехода не существует, ретраить смысла нет
            retry = False
        except KnownStartrekError as exc:
            error_message = exc.message
        except StartrekError:
            error_message = TransitionFailed.message

        if error_message:
            if retry:
                operation.delay(**fields)
            self.raise_error(error_message, fail_silently=retry)

    def update_ticket(self, **fields):
        if waffle.switch_is_active('ignore_job_issue_workflow'):
            return
        operation = IssueUpdateOperation(self.instance.startrek_key)
        try:
            operation(**fields)
        except StartrekError:
            operation.delay(**fields)


class BudgetPositionMixin:

    def get_and_check_budget_position(self, bp_id, date=None):
        """
        Обертка над функцией get_budget_position, которая проверяет статус
        """
        bp = get_budget_position(bp_id, date)
        if bp.get('hiring') != BUDGET_POSITION_STATUSES.vacancy:
            logger.error('Budget position %s is not in VACANCY status', bp_id)
            raise BudgetPositionError('budget_position_invalid_status')
        return bp


class ApproveAction(BudgetPositionMixin, StartrekMixin, VacancyActionBase):

    valid_statuses = (
        VACANCY_STATUSES.on_approval,
    )

    def perform(self, **params):
        bp_id = params['budget_position_id']
        bp = self.get_and_check_budget_position(bp_id)

        if not self.check_startrek_issue_status({StatusEnum.in_progress}):
            raise WorkflowError(IssueInvalidStatus.message)

        self.instance.budget_position_id = bp_id
        self.instance.set_main_recruiter(params['main_recruiter'])
        for recruiter in params['recruiters']:
            self.instance.add_recruiter(recruiter)
        self.instance.status = VACANCY_STATUSES.in_progress
        self.instance.save()

        context = self.get_context()
        context['bp'] = bp
        self.update_ticket(
            assignee=params['main_recruiter'].username,
            recruiter=params['main_recruiter'].username,
            comment=loader.render_to_string(
                template_name='startrek/vacancies/approve.wiki',
                context=context,
            ),
        )

        notification = VacancyApprovedNotification(self.instance, self.user)
        notification.send()

        vacancy_approved.send(Vacancy, vacancy=self.instance)

        return self.instance


class CloseAction(StartrekMixin, VacancyActionBase):

    valid_statuses = (
        VACANCY_STATUSES.suspended,
        VACANCY_STATUSES.on_approval,
        VACANCY_STATUSES.in_progress,
    )

    def create_registry_transaction(self):
        if self.instance.budget_position_id:
            bp_transaction_data = BPRegistryVacancyCancellationSerializer(self.instance).data
            try:
                BPRegistryAPI.create_transaction(bp_transaction_data)
            except BPRegistryError:
                logger.exception('Failed to create transaction for vacancy cancellation')

    def perform(self, **params):
        was_on_approval = self.instance.status == VACANCY_STATUSES.on_approval
        if was_on_approval:
            self.instance.resolution = VACANCY_RESOLUTIONS.unapproved
        else:
            self.instance.resolution = VACANCY_RESOLUTIONS.cancelled
        self.instance.status = VACANCY_STATUSES.closed
        self.instance.save()

        archive_vacancy_publications(self.instance)

        self.create_registry_transaction()

        if params.get('application_resolution'):
            close_vacancy_applications(
                vacancy=self.instance,
                resolution=params['application_resolution'],
                initiator=self.user,
            )

        context = self.get_context()
        context['comment'] = params.get('comment')
        comment = loader.render_to_string('startrek/vacancies/close.wiki', context)

        is_autohire = self.instance.type == VACANCY_TYPES.autohire
        if not self.is_startrek_operations_disabled():
            operation = IssueTransitionOrUpdateOperation(self.instance.startrek_key)
            transition = TransitionEnum.declined if is_autohire else TransitionEnum.cancel
            operation.delay(transition, comment=comment)
        if not was_on_approval and not is_autohire:
            notification = VacancyClosedNotification(self.instance, self.user)
            notification.send()

        return self.instance


class CloseByBPRegistryAction(CloseAction):

    def has_permission(self):
        return self.user.has_perm('permissions.can_use_api_for_staff')  # robot-staff

    def create_registry_transaction(self):
        pass

    def perform(self, **params):
        params['application_resolution'] = APPLICATION_RESOLUTIONS.vacancy_closed
        return super().perform(**params)


class SuspendAction(StartrekMixin, VacancyActionBase):

    valid_statuses = (
        VACANCY_STATUSES.in_progress,
    )

    def perform(self, **params):
        self.instance.status = VACANCY_STATUSES.suspended
        self.instance.save()

        if params.get('need_close_applications'):
            close_vacancy_applications(
                vacancy=self.instance,
                resolution=params['application_resolution'],
                initiator=self.user,
            )

        context = self.get_context()
        context['comment'] = params.get('comment')
        self.change_ticket_status(
            transition=TransitionEnum.on_hold,
            comment=loader.render_to_string('startrek/vacancies/suspend.wiki', context),
        )

        notification = VacancySuspendedNotification(self.instance, self.user)
        notification.send()

        return self.instance


class ResumeAction(BudgetPositionMixin, StartrekMixin, VacancyActionBase):

    valid_statuses = (
        VACANCY_STATUSES.suspended,
    )

    def perform(self, **params):
        self.get_and_check_budget_position(self.instance.budget_position_id)

        self.instance.status = VACANCY_STATUSES.in_progress
        self.instance.save()

        context = self.get_context()
        context['comment'] = params.get('comment')
        self.change_ticket_status(
            transition=TransitionEnum.start_progress,
            comment=loader.render_to_string('startrek/vacancies/resume.wiki', context),
        )

        notification = VacancyResumedNotification(self.instance, self.user)
        notification.send()

        return self.instance


class UpdateAction(StartrekMixin, Action):

    valid_statuses = (
        VACANCY_STATUSES.draft,
        VACANCY_STATUSES.on_approval,
        VACANCY_STATUSES.in_progress,
        VACANCY_STATUSES.suspended,
        VACANCY_STATUSES.offer_processing,
        VACANCY_STATUSES.offer_accepted,
    )

    def has_permission(self):
        return (
            self.user.is_recruiter
            or self.user in self.instance.team
        )

    def perform(self, **params):
        old_data = VacancyDiffEstimationSerializer(self.instance).data
        old_main_recruiter = self.instance.main_recruiter

        self.instance = update_or_create_vacancy(
            data=params,
            initiator=self.user,
            instance=self.instance,
        )

        updated_instance = (
            Vacancy.unsafe
            .select_related(
                'department',
                'profession',
                'professional_sphere',
            )
            .prefetch_related(
                'memberships__member',
            )
            .get(pk=self.instance.pk)
        )

        new_main_recruiter = updated_instance.main_recruiter
        if old_main_recruiter != new_main_recruiter and updated_instance.startrek_key:
            comment = loader.render_to_string(
                template_name='startrek/vacancies/main-recruiter-changed.wiki',
                context={
                    'user': new_main_recruiter,
                },
            )
            issue_data = {
                'recruiter': new_main_recruiter.username,
                'comment': comment,
            }

            # Для вакансий, у которых JOB-тикет в работе или приостановлен,
            # меняем так же и исполнителя
            if self.check_startrek_issue_status({StatusEnum.in_progress, StatusEnum.on_hold}):
                issue_data['assignee'] = new_main_recruiter.username

            self.update_ticket(**issue_data)

        new_data = VacancyDiffEstimationSerializer(updated_instance).data
        diff = VacancyDiffEstimationSerializer.get_diff(old_data, new_data)

        if diff:
            notification = VacancyUpdatedNotification(self.instance, self.user, diff=diff)
            notification.send()
        return self.instance


class AddToGroupAction(VacancyActionBase):

    valid_statuses = (
        VACANCY_STATUSES.in_progress,
        VACANCY_STATUSES.offer_processing,
    )

    def perform(self, **params):
        params['vacancy_group'].vacancies.add(self.instance)
        return self.instance


class UpdateByStaffAction(Action):

    valid_statuses = (
        VACANCY_STATUSES.suspended,
        VACANCY_STATUSES.in_progress,
        VACANCY_STATUSES.on_approval,
        VACANCY_STATUSES.offer_processing,
        VACANCY_STATUSES.offer_accepted,
    )

    valid_offer_statuses = (
        OFFER_STATUSES.sent,
        OFFER_STATUSES.accepted,
    )

    def is_visible(self):
        return False

    def is_available(self):
        return True

    @cached_property
    def offer(self):
        return self.instance.offers.alive().first()

    def is_status_correct(self):
        if not super().is_status_correct():
            return False

        return self.offer is None or self.offer.status in self.valid_offer_statuses

    def _change_value_stream(self, new_value_stream, context):
        data = {'value_stream': new_value_stream}
        if self.instance.value_stream == new_value_stream:
            return

        context['is_value_stream_changed'] = True
        context['old_vacancy_value_stream'] = self.instance.value_stream

        self.instance = update_or_create_vacancy(
            data=data,
            initiator=self.user,
            instance=self.instance,
        )

    def _change_geography(self, new_geography, context):
        from intranet.femida.src.offers.controllers import OfferCtl  # FIXME

        data = {'geography': new_geography}
        if self.instance.geography != new_geography:
            context['is_geography_changed'] = True
            context['old_vacancy_geography'] = self.instance.geography

            self.instance = update_or_create_vacancy(
                data=data,
                initiator=self.user,
                instance=self.instance,
            )

        if self.offer and self.offer.geography != new_geography:
            context['is_geography_changed'] = True
            context['old_offer_geography'] = self.offer.geography

            offer_ctl = OfferCtl(self.offer)

            offer_ctl.update(data)

    def _change_department(self, new_department, context):
        from intranet.femida.src.offers.controllers import OfferCtl  # FIXME

        data = {'department': new_department}
        if self.instance.department != new_department:
            context['is_department_changed'] = True
            context['old_vacancy_department'] = self.instance.department

            self.instance = update_or_create_vacancy(
                data=data,
                initiator=self.user,
                instance=self.instance,
            )

        if self.offer and self.offer.department != new_department:
            context['is_department_changed'] = True
            context['old_offer_department'] = self.offer.department

            offer_ctl = OfferCtl(self.offer)

            offer_ctl.update(data)

            if self.offer.newhire_id:
                offer_ctl.update_in_newhire(data, sync=False)

            comment = loader.render_to_string('startrek/offers/department-changed.wiki', context)
            fields = (
                'startrek_hr_key',
                'startrek_adaptation_key',
                'startrek_bootcamp_key',
                'startrek_relocation_key',
                'startrek_signup_key',
                'startrek_bonus_key',
                'startrek_eds_key',
            )
            for field in fields:
                key = getattr(self.offer, field)
                if key:
                    IssueCommentOperation(key).delay(comment)

            # Обновляем подразделение в Я.Найме
            if self.offer.oebs_person_id:
                update_oebs_assignment_task.delay(self.offer.id)

    def perform(self, **params):
        context = self.get_context()
        context['is_department_changed'] = False
        context['is_value_stream_changed'] = False
        context['is_geography_changed'] = False
        context.update(params)

        if not self.is_status_correct():
            logger.info(
                'Vacancy status %s, offer %s',
                self.instance.status,
                self.offer and self.offer.status,
            )
            raise WorkflowError('incorrect_vacancy_status')

        if not self.has_permission():
            raise WorkflowError('no_permissions')

        department = params.get('department')
        value_stream = params.get('value_stream')
        geography = params.get('geography')

        if department:
            self._change_department(department, context)
        if value_stream:
            self._change_value_stream(value_stream, context)
        if geography:
            self._change_geography(geography, context)

        should_comment_to_job_issue = (
            self.instance.startrek_key
            and (
                context['is_department_changed']
                or context['is_value_stream_changed']
                or context['is_geography_changed']
            )
        )
        if should_comment_to_job_issue:
            comment = loader.render_to_string('startrek/vacancies/update-by-staff.wiki', context)
            operation = IssueUpdateOperation(self.instance.startrek_key)
            issue_params = {
                'comment': comment,
                'productHr2': self.instance.value_stream and self.instance.value_stream.startrek_id,
                'geography2': self.instance.geography and self.instance.geography.startrek_id,
            }
            if context.get('old_vacancy_department') or context.get('old_offer_department'):
                issue_params['tags'] = {'add': [settings.STARTREK_JOB_DEPARTMENT_CHANGE_TAG]}

            operation.delay(**issue_params)

        return self.instance


class VacancyWorkflow(Workflow):

    ACTION_MAP = {
        'approve': ApproveAction,
        'add_to_group': AddToGroupAction,
        'close': CloseAction,
        'close_by_bp_registry': CloseByBPRegistryAction,
        'suspend': SuspendAction,
        'resume': ResumeAction,
        'update': UpdateAction,
        'update_by_staff': UpdateByStaffAction,
    }
