import logging

from django.utils.html import escape

from intranet.femida.src.wf.base import wiki_format
from intranet.femida.src.wf.tasks import wiki_reformat_instance


logger = logging.getLogger(__name__)


def get_safe_wiki_method(field_name):
    """
    Для вики-поля отдает метод, который отдает html:
      - если formatted_comment существует, то его
      - если не существует, то саму разметку, обернутую в простой html
    """
    def method(self):
        field = getattr(self, field_name)
        wiki_field = getattr(self, self.WIKI_FIELDS_MAP[field_name])

        if field and not wiki_field:
            return '<div>{}</div>'.format(escape(field).replace('\n', '<br>'))
        else:
            return wiki_field

    return method


class WFModelMixin:
    """
    Миксин для модели, в который спрятана логика форматирования wiki-полей.

    WIKI_FIELDS_MAP - словарь, ключи которого названия полей с вики-разметкой,
    а значения - поля с полученным html от WF.

    _snapshot_map - маппинг вики-полей на старые значения, до того, как мы
    что-то поредактировали. Нужен для того, чтобы узнать, нужно ли вообще
    wiki-форматирвоание.

    TODO: Было бы неплохо создавать поля formatted_ автоматически
    """
    WIKI_FIELDS_MAP = {}

    def __init__(self, *args, **kwargs):
        self._snapshot_map = {}
        super().__init__(*args, **kwargs)
        for field_name, wiki_field_name in self.WIKI_FIELDS_MAP.items():
            self._snapshot_map[field_name] = getattr(self, field_name)

            # Для каждого wiki-поля добавляем парный ему метод, который будет отдавать
            # html на основе разметки, даже если не вышло сходить в wf
            method = get_safe_wiki_method(field_name)
            setattr(self, 'get_safe_' + wiki_field_name, method.__get__(self, self.__class__))

    def format_wiki_fields(self, force=False, timeout=None, save=False):
        """
        :param force: Если True, то форматируем wiki-поля даже если разметка не менялась
        :param timeout: таймаут вики-форматтера
        :param save: Если True, то вызываем save модели
        :return: bool - успешное форматирование
        """
        is_ok = True
        update_fields = []
        for field_name, wiki_field_name in self.WIKI_FIELDS_MAP.items():
            new_value = getattr(self, field_name)
            if force or new_value != self._snapshot_map.get(field_name):
                try:
                    formatted = wiki_format(text=new_value, timeout=timeout, safe=False)
                except Exception:
                    is_ok = False
                    formatted = ''
                setattr(self, wiki_field_name, formatted)
                update_fields.append(wiki_field_name)
        if save:
            self.save(ignore_wiki_format=True, update_fields=update_fields)
        if not is_ok and self.id:
            logger.error('Error during format %s with id %d', self.__class__.__name__, self.id)
        return is_ok

    def save(self, *args, **kwargs):
        """
        force_wiki_format - при сохранении всегда пытаемся отформатировать wiki-поля
        ignore_wiki_format - при сохранении игнорируем переформатирование wiki-полей
        """
        force_insert = kwargs.get('force_insert')
        force_wiki_format = kwargs.pop('force_wiki_format', False)
        ignore_wiki_format = kwargs.pop('ignore_wiki_format', False)
        retry_wiki_format = False
        is_new_instance = self.id is None

        if force_wiki_format and ignore_wiki_format:
            raise ValueError

        if not ignore_wiki_format:
            force = force_wiki_format or force_insert
            retry_wiki_format = not self.format_wiki_fields(force=force)

        # Если в save передали update_fields, то мы должны убедиться, что вместе
        # с полем для вики-разметки будет сохранено поле для html
        if kwargs.get('update_fields') is not None:
            for k, v in self.WIKI_FIELDS_MAP.items():
                if k in kwargs['update_fields']:
                    kwargs['update_fields'].append(v)

        super().save(*args, **kwargs)

        # Форматируем отложено в случае неудачи
        if retry_wiki_format:
            if is_new_instance:
                logger.error('Error during format %s with id %d', self.__class__.__name__, self.id)
            wiki_reformat_instance.delay(
                app_label=self._meta.app_label,
                model_name=self._meta.model_name,
                instance_id=self.id,
            )

        # Обновляем снапшоты wiki-полей
        for field_name in self.WIKI_FIELDS_MAP:
            self._snapshot_map[field_name] = getattr(self, field_name)
