import sys
import logging
from collections import defaultdict
from datetime import datetime, timedelta
import yt.wrapper as yt

from django.core.management.base import BaseCommand
from django.conf import settings
from django.utils import timezone

from intranet.femida.src.yt import dummy
from intranet.femida.src.yt.jobs.common import get_rows_by_yandexuids


logger = logging.getLogger(__name__)


class YtJobCommand(BaseCommand):

    # Кодировка, через которую по умолчанию будут кодироваться строки в YT
    yson_encoding = 'utf-8'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._setup_yt_config()

    def _setup_yt_config(self):
        yt.config['token'] = settings.YT_TOKEN
        yt.config['proxy']['url'] = settings.YT_PROXY

        # "Обнуляем" __main__ модуль, иначе будем ловить всякие ImportError
        sys.modules['__main__'] = dummy


class SubmissionsYtJobCommand(YtJobCommand):
    """
    Базовая команда, которая достает yandexuid из откликов и забирает данные из YT по этим uid
    """
    command_code = None

    def add_arguments(self, parser):
        parser.add_argument('--submission_id')
        parser.add_argument('--from_dt')

    def process_rows(self, rows):
        return

    def get_input_table(self):
        return

    def handle(self, *args, **kwargs):
        # Если вынести импорт наверх, то джобы почему-то не выполняются
        from intranet.femida.src.candidates.models import CandidateSubmission
        from intranet.femida.src.candidates.choices import EXTERNAL_SUBMISSION_SOURCES

        base_qs = CandidateSubmission.unsafe.filter(source__in=EXTERNAL_SUBMISSION_SOURCES._db_values)

        submission_id = kwargs.get('submission_id')
        if submission_id:
            self.submissions = base_qs.filter(id=submission_id)
        else:
            from_dt = kwargs.get('from_dt')
            if from_dt is not None:
                from_dt = datetime.strptime(from_dt, '%Y-%m-%d')
            else:
                from_dt = timezone.now() - timedelta(days=1)

            self.submissions = base_qs.filter(created__gte=from_dt)

        # Словарь, где ключ - yandexuid, а значение - список откликов
        # При этом yandexuid мы приводим к int, потому что в YT-таблицах он хранится как int
        self.uid_submissions_map = defaultdict(list)
        submissions_count = 0
        for s in self.submissions:
            yandexuid = s.yuid

            if not yandexuid:
                continue

            try:
                yandexuid = int(yandexuid)
            except ValueError:
                logger.warning('[%s] Broken yandexuid: %s', self.command_code, yandexuid)
            else:
                submissions_count += 1
                self.uid_submissions_map[yandexuid].append(s)

        logger.info('[%s] %d submissions to process', self.command_code, submissions_count)
        if not self.uid_submissions_map:
            return

        rows = get_rows_by_yandexuids(
            input_table=self.get_input_table(),
            yandexuids=set(self.uid_submissions_map.keys()),
            yson_encoding=self.yson_encoding,
        )
        self.process_rows(rows)
