from django.conf import settings
from rest_framework import serializers

from intranet.femida.src.candidates.models import CandidateAttachment, Verification
from intranet.femida.src.core.serializers import TimeStampField
from intranet.femida.src.interviews.models import Interview, InterviewRound, InterviewRoundTimeSlot
from intranet.femida.src.offers.models import Offer


class CandidateAttachmentYTSerializer(serializers.ModelSerializer):

    uploader = serializers.CharField(source='attachment.uploader.username', default=None)
    responsibles = serializers.SerializerMethodField()
    attachment_name = serializers.ReadOnlyField(source='attachment.name', default=None)
    sha1 = serializers.ReadOnlyField(source='attachment.sha1', default=None)
    mime_type = serializers.CharField(source='attachment.content_type', default=None)
    text = serializers.CharField(source='attachment.text', default=None)
    professions = serializers.SerializerMethodField()
    full_name = serializers.ReadOnlyField(source='candidate.get_full_name')
    mds = serializers.CharField(source='attachment.attached_file.url', default=None)

    def get_responsibles(self, obj):
        return [user.username for user in obj.candidate.responsibles.all()]

    def get_professions(self, obj):
        return [c.profession_id for c in obj.candidate.candidate_professions.all()]

    class Meta:
        model = CandidateAttachment
        yt_schema = {
            'uploader': 'string',
            'responsibles': 'any',
            'candidate_id': 'int32',
            'attachment_id': 'int32',
            'attachment_name': 'string',
            'sha1': 'string',
            'mime_type': 'string',
            'text': 'string',
            'professions': 'any',
            'full_name': 'string',
            'mds': 'string',
        }
        fields = tuple(yt_schema)


class VerificationYTSerializer(serializers.ModelSerializer):

    recruiter = serializers.CharField(source='created_by.username', default=None)
    verification_id = serializers.IntegerField(source='id')
    full_name = serializers.ReadOnlyField(source='candidate.get_full_name')
    phone_numbers = serializers.SerializerMethodField()
    emails = serializers.SerializerMethodField()
    cv_id = serializers.IntegerField(source='attachment.id', default=None)
    cv_content_type = serializers.CharField(source='attachment.content_type', default=None)
    cv_text = serializers.CharField(source='attachment.text', default=None)
    cv_mds = serializers.CharField(source='attachment.attached_file', default=None)
    profession_id = serializers.IntegerField(
        source='application.vacancy.profession_id',
        default=None,
    )
    yandexuid = serializers.CharField(source='raw_data.params.yandexuid', default=None)
    department_url = serializers.CharField(
        source='application.vacancy.department.url',
        default=None,
    )

    def get_phone_numbers(self, obj):
        return [c.normalized_account_id for c in obj.candidate.contacts.all() if c.type == 'phone']

    def get_emails(self, obj):
        return [c.normalized_account_id for c in obj.candidate.contacts.all() if c.type == 'email']

    class Meta:
        model = Verification
        yt_schema = {
            'recruiter': 'string',
            'verification_id': 'int32',
            'candidate_id': 'int32',
            'full_name': 'string',
            'phone_numbers': 'any',
            'emails': 'any',
            'cv_id': 'int32',
            'cv_content_type': 'string',
            'cv_text': 'string',
            'cv_mds': 'string',
            'profession_id': 'int32',
            'yandexuid': 'string',
            'department_url': 'string',
        }
        fields = tuple(yt_schema)


class OfferYTSerializer(serializers.ModelSerializer):

    timestamp = TimeStampField()
    offer_id = serializers.IntegerField(source='id')
    passport_pages = serializers.SerializerMethodField()
    snils = serializers.SerializerMethodField()
    residence_address = serializers.CharField(source='profile.residence_address')

    def get_passport_pages(self, obj):
        return [a.attached_file.url for a in obj.passport_pages]

    def get_snils(self, obj):
        if obj.snilses:
            return obj.snilses[0].attached_file.url

    class Meta:
        model = Offer
        yt_schema = {
            'timestamp': 'string',
            'offer_id': 'int32',
            'docs_request_count': 'int32',
            'candidate_id': 'int32',
            'full_name': 'string',
            'passport_pages': 'any',
            'snils': 'string',
            'residence_address': 'string',
            'join_at': 'string',
        }
        fields = tuple(yt_schema)


class InterviewSerializer(serializers.ModelSerializer):

    potential_interviewers = serializers.SerializerMethodField()

    def get_potential_interviewers(self, obj):
        return [i.username for i in obj.potential_interviewers.all()]

    class Meta:
        model = Interview
        fields = (
            'id',
            'type',
            'aa_type',
            'potential_interviewers',
        )


class InterviewRoundTimeSlotSerializer(serializers.ModelSerializer):

    class Meta:
        model = InterviewRoundTimeSlot
        fields = (
            'start',
            'end',
        )


class InterviewRoundYTSerializer(serializers.ModelSerializer):

    timestamp = TimeStampField()
    recruiter = serializers.ReadOnlyField(source='created_by.username')
    office = serializers.ReadOnlyField(source='office.name_ru', default=None)
    time_slots = InterviewRoundTimeSlotSerializer(many=True)
    interviews = InterviewSerializer(many=True)

    class Meta:
        model = InterviewRound
        yt_schema = {
            'timestamp': 'string',
            'id': 'int32',
            'candidate_id': 'int32',
            'recruiter': 'string',
            'office': 'string',
            'time_slots': 'any',
            'interviews': 'any',
            'is_strict_order': 'boolean',
            'lunch_duration': 'int32',
            'comment': 'string',
        }
        fields = tuple(yt_schema)


class ScreenshotYTSerializer(serializers.Serializer):

    # https://wiki.yandex-team.ru/vanadium/batch-format/#formatvxodnojjtablicy

    url = serializers.SerializerMethodField()
    meta = serializers.SerializerMethodField()

    def get_url(self, obj):
        return f'{settings.FEMIDA_PROTOCOL}://{settings.FEMIDA_HOST}/publications/{obj.uuid}/og/'

    def get_meta(self, obj):
        return {'publication_id': obj.id}

    class Meta:
        yt_schema = {
            'url': 'string',
            'meta': 'any',
        }


class ScoringUploadResultSerializer(serializers.Serializer):

    timestamp = TimeStampField()
    message = serializers.SerializerMethodField()

    def get_message(self, obj):
        return str(obj['message'])

    class Meta:
        yt_schema = {
            'timestamp': 'string',
            'message': 'string',
        }
