import waffle

from datetime import timedelta

from django.db import transaction
from django.utils import timezone

from intranet.femida.src.actionlog.decorators import action_logged
from intranet.femida.src.candidates.models import CandidateAttachment, Verification
from intranet.femida.src.celery_app import app
from intranet.femida.src.interviews.choices import (
    INTERVIEW_ROUND_PLANNERS,
    INTERVIEW_ROUND_STATUSES,
)
from intranet.femida.src.interviews.models import InterviewRound
from intranet.femida.src.interviews.tasks import interview_round_finish_planning_task
from intranet.femida.src.offers.models import Offer
from intranet.femida.src.offers.yang.helpers import get_yang_unprocessed_doc_fields
from intranet.femida.src.stats.utils import (
    get_beginning_of_moscow_day as get_midnight,
    str_to_moscow_date,
)
from intranet.femida.src.utils.lock import locked_task
from intranet.femida.src.yt.base import (
    AttachmentYTTable,
    InterviewRoundYTTable,
    OfferYTTable,
    ProcessedInterviewRoundYTTable,
    ProcessedOfferYTTable,
    VerificationYTTable,
)


@app.autoretry_task(max_retries=3)
@locked_task
def save_attachments_in_yt(to_date: str = None):
    if to_date:
        to_date = str_to_moscow_date(to_date) + timedelta(days=1)
    else:
        to_date = get_midnight(timezone.now())
    from_date = to_date - timedelta(days=1)
    qs = CandidateAttachment.unsafe.filter(
        created__gte=from_date,
        created__lt=to_date,
    )
    table = AttachmentYTTable(from_date)
    table.write(qs)


@app.autoretry_task(max_retries=3)
@locked_task
def save_verifications_in_yt(to_date: str = None):
    if to_date:
        to_date = str_to_moscow_date(to_date) + timedelta(days=1)
    else:
        to_date = get_midnight(timezone.now())
    from_date = to_date - timedelta(days=1)
    qs = Verification.objects.filter(
        created__gte=from_date,
        created__lt=to_date,
    )
    table = VerificationYTTable(from_date)
    table.write(qs)


@app.autoretry_task(max_retries=3)
def save_offer_data_in_yt(offer_id: int):
    """
    Создаем в YT-таблице запись с полями на распознавание асессорами

    Описание процесса оформления в кадрах асессорами:
    https://wiki.yandex-team.ru/intranet/femida/dev/yang-hr/
    """
    document_fields = {'passport_pages', 'snils', 'residence_address'}

    offer_qs = Offer.unsafe.filter(id=offer_id)
    exclude = document_fields - get_yang_unprocessed_doc_fields(offer_qs.first())
    table = OfferYTTable(exclude=exclude)
    table.write(offer_qs, append=True)


@app.autoretry_task(max_retries=3)
@locked_task
def get_processed_offer_data_from_yt():
    from intranet.femida.src.offers.yang.controllers import update_offers_from_yt
    if not waffle.switch_is_active('enable_yang_results_fetching'):
        return
    table = ProcessedOfferYTTable()
    raw_data = table.read()
    update_offers_from_yt(raw_data)


@app.autoretry_task(max_retries=3)
@transaction.atomic
@action_logged('interview_round_start_planning')
def save_interview_round_data_in_yt(interview_round_id: int):
    qs = InterviewRound.objects.filter(id=interview_round_id)
    table = InterviewRoundYTTable()
    table.write(qs, append=True)

    interview_round = qs.first()
    interview_round.planner = INTERVIEW_ROUND_PLANNERS.yang
    interview_round.status = INTERVIEW_ROUND_STATUSES.planning
    interview_round.save(update_fields=('planner', 'status', 'modified'))


@app.autoretry_task(max_retries=3)
@locked_task
def get_processed_interview_round_data_from_yt():
    planning_round_ids = set(
        InterviewRound.objects
        .filter(
            planner=INTERVIEW_ROUND_PLANNERS.yang,
            status=INTERVIEW_ROUND_STATUSES.planning,
        )
        .values_list('id', flat=True)
    )
    table = ProcessedInterviewRoundYTTable()
    data = (row for row in table.read() if row['id'] in planning_round_ids)
    for row in data:
        interview_round_finish_planning_task.delay(row['id'], row)


@app.autoretry_task(max_retries=3)
def save_screenshot_data_in_yt(publication_ids):
    from intranet.femida.src.publications.models import Publication
    from intranet.femida.src.yt.base import ScreenshotYTTable

    qs = Publication.objects.filter(id__in=publication_ids)
    table = ScreenshotYTTable()
    table.write(qs)

    return table.path
