# -*- coding: utf-8 -*-
import json
import os
import yaml
import yenv

from pydantic import BaseSettings, BaseModel
from typing import List, Optional

from .types import EnvTypes, AppTypes


class YamlValue:
    def __inst__(self):
        self.default = None
        self.value = None

    def __getitem__(self, app_type: AppTypes):
        if isinstance(self.value, dict) and '.app_type' in self.value:
            return self.value['.app_type'].get(app_type.value, self.default)
        return self.value or self.default

    @staticmethod
    def create(data, env_type) -> 'YamlValue':
        inst = YamlValue()
        if isinstance(data, dict) and '.env_type' in data:
            inst.default = data.get('default')
            inst.value = data['.env_type'].get(env_type.value)
        else:
            inst.value = data
        return inst


class YamlConfig:
    def __init__(self):
        self._environ = self._get_environ()
        self.fields = {}

    @staticmethod
    def create(config_name: str, secret_name: str, env_type: EnvTypes) -> 'YamlConfig':
        inst = YamlConfig()
        for file_name in (config_name, secret_name):
            content = YamlConfig._get_content(file_name)
            config = yaml.safe_load(content)
            for key, value in config.items():
                value = inst._get_environ_value(key, value, env_type)
                inst.fields[key] = value
        return inst

    @staticmethod
    def _get_content(content_name):
        if content_name.startswith('arcadia:'):
            from library.python.resource import resfs_read
            return resfs_read(content_name[8:]).decode()
        else:
            with open(content_name) as f:
                return f.read()

    def _get_environ(self):
        return {
            key.lower(): value
            for key, value in os.environ.items()
        }

    def _get_environ_value(self, key, value, env_type: EnvTypes):
        key_lower = key.lower()
        if key_lower in self._environ:
            value = self._environ[key]
            try:
                value = json.loads(value)
            except json.JSONDecodeError:
                pass
        return YamlValue.create(value, env_type)


class AppTypeSettings(BaseModel):
    app_type: AppTypes

    database_hosts: List[str] = []
    database_port: int = 6432
    database_name: str = ''
    database_user: str = ''

    redis_hosts: List[str] = []
    redis_port: int = 26379
    redis_db: int = 0
    redis_cluster: str = ''

    blackbox_name: str = 'ProdYateam'

    async_tvm2_class_name: str = 'intranet.forms.micro.core.src.http.MockAsyncTVM2'
    tvm2_class_name: str = 'intranet.forms.micro.core.src.http.MockTVM2'

    tvm2_client_id: int = 2001259
    tvm2_allowed_client_ids: List[int] = []
    tracker_tvm2_client_id: int = 177
    wiki_tvm2_client_id: int = 2002678
    sender_tvm2_client_id: int = 2000074
    zora_tvm2_client_id: int = 2000193
    webmaster_tvm2_client_id: int = 2000286
    blackbox_tvm2_client_ids: List[int] = [222, 223, 239]

    database_password: str = ''
    redis_password: str = ''
    tvm2_client_secret: str = ''
    elliptics_token: str = ''
    http_integration_oauth_token: str = ''
    survey_id_salt: bytes = b''

    def get_tvm2_destinations(self):
        return [
            self.tracker_tvm2_client_id,
            self.wiki_tvm2_client_id,
            self.sender_tvm2_client_id,
            self.zora_tvm2_client_id,
            self.webmaster_tvm2_client_id,
        ] + self.blackbox_tvm2_client_ids


class GlobalSettings(BaseSettings):
    env_type: EnvTypes = yenv.type
    default_app_type: AppTypes = AppTypes.forms_int
    config_name: str = 'arcadia:intranet/forms/micro/config.yaml'
    secret_name: str = 'arcadia:intranet/forms/micro/secret.yaml'
    uvicorn_port: int = 80
    uvicorn_loglevel: str = 'info'
    uvicorn_reload: bool = False
    pretty_logs: bool = False

    forms_int: Optional[AppTypeSettings]
    forms_ext: Optional[AppTypeSettings]
    forms_biz: Optional[AppTypeSettings]

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        yaml_config = YamlConfig.create(self.config_name, self.secret_name, self.env_type)
        for app_type in AppTypes:
            setattr(self, app_type.name, self._get_settings(app_type, yaml_config))

    def _get_settings(self, app_type: AppTypes, yaml_config: YamlConfig) -> AppTypeSettings:
        values = {
            key: yaml_config.fields[key][app_type]
            for key in yaml_config.fields
        }
        return AppTypeSettings(app_type=app_type, **values)

    def get_settings(self, app_type: AppTypes) -> AppTypeSettings:
        return getattr(self, app_type.name, None)


global_settings = GlobalSettings()
