# -*- coding: utf-8 -*-
import lazy_object_proxy

from pyscopg2.aiopg_sa import PoolManager

from .types import AppTypes
from .conf import global_settings


def get_database_host_string(hosts, port):
    return ','.join(f'{fqdn}:{port}' for fqdn in hosts)


def get_database_dsn(config):
    return '{engine}://{user}:{password}@{host}/{name}'.format(
        engine=config['engine'],
        user=config['user'],
        password=config['password'],
        host=config['host'],
        name=config['name'],
    )


def get_database_config(app_type: AppTypes):
    settings = global_settings.get_settings(app_type)
    return {
        'engine': 'postgresql',
        'host': get_database_host_string(settings.database_hosts, settings.database_port),
        'name': settings.database_name,
        'user': settings.database_user,
        'password': settings.database_password,
    }


class ApplicationPoolManager(PoolManager):
    def __init__(self, *args, **kwargs):
        self.app_type = kwargs.pop('app_type', AppTypes.forms_int)
        super().__init__(*args, **kwargs)

    @property
    def is_business(self):
        return self.app_type == AppTypes.forms_biz


class ApplicationFactory:
    def __init__(self, app_type: AppTypes):
        database_config = get_database_config(app_type)
        self._pool = ApplicationPoolManager(app_type=app_type, dsn=get_database_dsn(database_config))

    @staticmethod
    def create(app_type: AppTypes) -> 'ApplicationFactory':
        return lazy_object_proxy.Proxy(lambda: ApplicationFactory(app_type))

    def get_pool(self) -> ApplicationPoolManager:
        return self._pool


class AppTypeFactory:
    def __init__(self):
        self._factories = {
            app_type: ApplicationFactory.create(app_type)
            for app_type in AppTypes
        }

    def get(self, app_type: AppTypes) -> ApplicationFactory:
        return self._factories.get(app_type)

    def __getitem__(self, app_type: AppTypes) -> ApplicationFactory:
        return self._factories[app_type]


app_type_factory = AppTypeFactory()


def get_pool(app_type: AppTypes) -> ApplicationPoolManager:
    factory = app_type_factory[app_type]
    return factory.get_pool()
