#!/usr/bin/env python3
# -*- coding: utf-8 -*-
import argparse
import json
import requests
import os
import sys
import tempfile
import textwrap

from datetime import datetime
from io import StringIO
from requests.auth import AuthBase


TANKER_URL = 'https://tanker.yandex-team.ru/api/v1'
TANKER_TOKEN = os.environ.get('TANKER_TOKEN', 'test')
PO_FILE_PROLOG = '''# This is autogenerated file for {language}.
msgid ""
msgstr ""
"Content-Transfer-Encoding: 8bit\\n"
"Content-Type: text/plain; charset=UTF-8\\n"

'''

class OAuth(AuthBase):
    auth_header = 'Authorization'

    def __init__(self, token):
        self.auth_value = f'OAuth {token}'

    def __call__(self, request):
        if self.auth_header not in request.headers:
            request.headers[self.auth_header] = self.auth_value
        return request


class TankerClient:
    def __init__(self, project):
        self.project = project
        self.auth = OAuth(TANKER_TOKEN)

    def _make_request(self, method, path, **kwargs):
        url = f'{TANKER_URL}{path}'
        response = requests.request(method, url, auth=self.auth, **kwargs)
        response.raise_for_status()
        if response.content:
            return response.json()

    def get_keys(self, keyset):
        path = f'/project/{self.project}/keyset/{keyset}/key/'
        result = self._make_request('get', path)
        return result['items']

    def modify_keys(self, keyset, create_keys=None, delete_keys=None, commit_message=None):
        path = f'/project/{self.project}/keyset/{keyset}/batch/'
        commit_message = commit_message or ('Keys updated at %s' % datetime.now())
        data = {
            'commit_message': commit_message,
        }
        if create_keys:
            data['update'] = {
                'action': 'CREATE',
                'process_expired_translations': True,
                'keys': create_keys,
            }
        if delete_keys:
            data['delete'] = {
                'keys': list(delete_keys),
            }
        if create_keys or delete_keys:
            return self._make_request('post', path, json=data)


def po_string(text, width=70):
    result = StringIO()
    if not text or len(text) > width:
        result.write('""\n')
    for s in textwrap.wrap(text, width, drop_whitespace=False):
        result.write(json.dumps(s, ensure_ascii=False))
        result.write('\n')
    return result.getvalue()


def generate_po(keys, language):
    st = StringIO()
    st.write(PO_FILE_PROLOG.format(language=language.upper()))
    for key in keys:
        msgid = key['name']
        translations = {
            lang.lower(): translation
            for lang, translation in key['translations'].items()
        }
        msgstr = translations.get(language, {}).get('payload', {}).get('singular_form', '')
        if msgstr:
            st.write(f'msgid {po_string(msgid)}')
            st.write(f'msgstr {po_string(msgstr)}')
            st.write('\n')
    return st.getvalue()


def write_po(keys, output_path, languages, file_name=None, dry_run=False):
    file_name = file_name or 'django.po'
    for language in languages:
        locale_path = os.path.join(output_path, language, 'LC_MESSAGES')
        if not dry_run and not os.path.exists(locale_path):
            os.makedirs(locale_path)
        file_path = os.path.join(locale_path, file_name)
        print(f'Write file {file_path}')
        if not dry_run:
            with open(file_path, 'w') as f:
                f.write(generate_po(keys, language))


def parse_po(file_name):
    source_reference = []
    msgid = StringIO()
    msgstr = StringIO()
    st = None

    with open(file_name) as f:
        for line in f:
            line = line.strip()
            if line.startswith('#: ') or line.startswith('#, '):
                _, text = line.split(' ', 1)
                source_reference.extend(text.split('\n'))
            elif line.startswith('#'):
                continue
            elif line.startswith('msgid '):
                _, text = line.split(' ', 1)
                st = msgid
                st.write(json.loads(text))
            elif line.startswith('msgstr '):
                _, text = line.split(' ', 1)
                st = msgstr
                st.write(json.loads(text))
            elif line.startswith('"'):
                st.write(json.loads(line))
            elif not line:
                yield msgid.getvalue(), msgstr.getvalue(), ' '.join(source_reference)
                source_reference = []
                msgid.seek(0)
                msgid.truncate()
                msgstr.seek(0)
                msgstr.truncate()
                st = None
    if st:
        yield msgid.getvalue(), msgstr.getvalue(), ' '.join(source_reference)


def collect_po(root_path):
    _, file_path = tempfile.mkstemp(prefix='django', suffix='.po')
    command = f'xgettext -L python -o {file_path} $(grep -Rl gettext {root_path})'
    print(f'Execute {command}')
    st = os.popen(command)
    output = st.read()
    rc = st.close()
    if rc is None:
        return file_path

    print(f'Command returned status {rc}\n\n{output}')
    sys.exit(1)


def prepare_translation(translation):
    data = dict(translation)
    if 'commit_id' in data:
        del data['commit_id']
    data['action'] = 'CREATE'
    return data


def copy_keysets(from_project, from_keyset, to_project, to_keyset):
    from_client = TankerClient(from_project)
    from_keys = {
        key['name']: key
        for key in from_client.get_keys(from_keyset)
    }
    from_names = set(from_keys.keys())

    to_client = TankerClient(to_project)
    to_keys = {
        key['name']: key
        for key in to_client.get_keys(to_keyset)
    }
    to_names = set(to_keys.keys())

    create_keys = []
    for key_name in from_names - to_names:
        key = from_keys[key_name]
        meta = key.get('meta') or {}
        data = {
            'name': key_name,
            'plural': meta.get('plural', False),
            'context': meta.get('context', {}),
            'translations': {
                lang: prepare_translation(translation)
                for lang, translation in key.get('translations', {}).items()
            },
        }
        create_keys.append(data)

    if create_keys:
        commit_message = f'Copy keys from {from_keyset}'
        to_client.modify_keys(to_keyset, create_keys=create_keys, commit_message=commit_message)


def modify_keys(client, keyset, create_keys=None, delete_keys=None, dry_run=False):
    create_data = [
        {
            'name': key_name,
            'plural': False,
            'translations': {
                'ru': {
                    'name': key_name,
                    'language': 'ru',
                    'status': 'TRANSLATED',
                    'payload': {
                        'singular_form': key_name,
                    },
                },
            },

        }
        for key_name in create_keys or []
    ]
    if dry_run:
        for key in create_keys or []:
            print(f'Create key "{key}"')
        for key in delete_keys or []:
            print(f'Delete key "{key}"')
    else:
        client.modify_keys(keyset, create_keys=create_data, delete_keys=delete_keys)


def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-N', '--dry-run', help='Dry run', action='store_true')
    parser.add_argument('-D', '--delete', help='Delete unused keys', action='store_true')
    parser.add_argument('-p', '--project', required=True, help='Tanker project')
    parser.add_argument('-k', '--keyset', required=True, help='Tanker keyset')
    parser.add_argument('-l', '--languages', default='ru,en', help='Comma separated list of languages')
    parser.add_argument('-s', '--search-path', default='.', help='Root directory for input files search')
    parser.add_argument('-o', '--output-path', default='locale', help='Root path to store locale files')
    return parser.parse_args()


def main():
    args = get_args()

    client = TankerClient(args.project)
    keyset = args.keyset
    languages = args.languages.split(',')

    file_path = collect_po(args.search_path)
    actual_keys = set()
    if file_path:
        for (msgid, _, _) in parse_po(file_path):
            if msgid:
                actual_keys.add(msgid)

    keys = client.get_keys(keyset)
    known_keys = {key['name'] for key in keys}

    create_keys = actual_keys - known_keys
    if args.delete:
        delete_keys = known_keys - actual_keys
    else:
        delete_keys = None
    modify_keys(client, keyset, create_keys=create_keys, delete_keys=delete_keys, dry_run=args.dry_run)

    keys = client.get_keys(keyset)
    write_po(keys, args.output_path, languages, file_name=f'{keyset}.po', dry_run=args.dry_run)


if __name__ == '__main__':
    main()
