# -*- coding: utf-8 -*-
from django.db import models
from django.db.models.functions import Length
from django.db.models.query import QuerySet

from events.abc.client import AbcServiceClient
from events.common_app.managers import ProxyToQuerySetMixin
from events.translation.managers import TranslationQuerySetMixin


class AbcServiceQuerySet(TranslationQuerySetMixin, QuerySet):
    def filter_by_fit_for_search_text(self, search_text):
        return (
            self.filter_translated_fields(search_text, 'name', 'slug')
            .order_by(Length('name'), 'name')
        )


class AbcServiceManager(ProxyToQuerySetMixin, models.Manager):
    def get_queryset(self):
        return AbcServiceQuerySet(self.model, using=self._db)

    def sync_one_item(self, item, key_to_md5sum):
        abc_id = item['abc_id']
        name = item['name']
        translations = item['translations']
        slug = item['slug']
        is_active = item['is_active']
        is_deleted = item['is_deleted']
        md5sum = self.model.calc_md5sum(name, translations, slug, is_active, is_deleted)

        if abc_id not in key_to_md5sum:
            self.model.objects.create(
                abc_id=abc_id,
                name=name,
                translations=translations,
                slug=slug,
                is_active=is_active,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )
        elif key_to_md5sum[abc_id] != md5sum:
            self.model.objects.filter(abc_id=abc_id).update(
                name=name,
                translations=translations,
                slug=slug,
                is_active=is_active,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )

    def sync_with_source(self):
        key_to_md5sum = dict(
            pair
            for pair in self.model.objects.values_list('abc_id', 'md5sum')
        )
        sync_client = AbcServiceClient()
        for item in sync_client.make_request():
            self.sync_one_item(item, key_to_md5sum)
