# -*- coding: utf-8 -*-
from django.contrib.auth.models import Group, Permission
from guardian.models import UserObjectPermission, GroupObjectPermission
from rest_framework import serializers

from events.accounts.models import User
from events.geobase_contrib.models import City
from events.surveyme.models import Survey, SurveyGroup
from events.rest_framework_contrib.serializers import InternalModelSerializerV2Mixin, FilterByDataSerializerMixin


class GroupSerializer(FilterByDataSerializerMixin, InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = Group
        fields = (
            'id',
            'name',
        )


class UserSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    groups = GroupSerializer(many=True, filter_by_data=True, read_only=False)

    class Meta:
        model = User
        fields = (
            'id',
            'email',
            'username',
            'groups',
            'is_staff',
            'is_superuser',
        )


class ProfileCitySerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = City
        fields = (
            'id',
            'full_name'
        )


class UserPrivateSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    email = serializers.CharField()
    login = serializers.CharField(source='username')
    uid = serializers.CharField()
    cloud_uid = serializers.CharField()
    is_staff = serializers.CharField()
    is_superuser = serializers.CharField()
    name_and_surname_with_fallback = serializers.CharField(source='get_name_and_surname_with_fallback', read_only=True)
    name_with_fallback = serializers.CharField(source='get_name_with_fallback', read_only=True)

    class Meta:
        model = User
        fields = (
            'id',
            'email',
            'login',
            'uid',
            'cloud_uid',
            'is_staff',
            'is_superuser',
            'name_and_surname_with_fallback',
            'name_with_fallback',
        )


class PermissionSerializer(InternalModelSerializerV2Mixin, FilterByDataSerializerMixin, serializers.ModelSerializer):
    content_type_id = serializers.IntegerField()

    class Meta:
        model = Permission
        fields = (
            'id',
            'codename',
            'content_type_id',
            'name',
        )


class SurveyUserObjectPermissionContentObjectSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = Survey
        fields = (
            'id',
            'name',
        )


class SurveyGroupUserObjectPermissionContentObjectSerializer(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    class Meta:
        model = SurveyGroup
        fields = (
            'id',
            'name',
        )


class UserObjectPermissionContentObjectField(serializers.RelatedField):
    def to_representation(self, value):
        if not value:
            return None
        elif isinstance(value, Survey):
            serializer = SurveyUserObjectPermissionContentObjectSerializer(value)
        elif isinstance(value, SurveyGroup):
            serializer = SurveyGroupUserObjectPermissionContentObjectSerializer(value)
        else:
            raise Exception('Unexpected type of permission content object')

        return serializer.data


class ObjectPermissionSerializerBase(InternalModelSerializerV2Mixin, serializers.ModelSerializer):
    content_type_id = serializers.IntegerField()
    object_pk = serializers.IntegerField()
    permission = PermissionSerializer()
    content_object = UserObjectPermissionContentObjectField(read_only=True, required=False)


class UserObjectPermissionSerializer(ObjectPermissionSerializerBase):
    class Meta:
        model = UserObjectPermission
        fields = (
            'id',
            'content_type_id',
            'object_pk',
            'permission',
            'content_object',
        )


class GroupObjectPermissionSerializer(ObjectPermissionSerializerBase):
    class Meta:
        model = GroupObjectPermission
        fields = (
            'id',
            'content_type_id',
            'object_pk',
            'permission',
            'content_object',
        )
