# -*- coding: utf-8 -*-
import blackbox

from django.conf import settings
from django.contrib.auth.models import Group
from django.contrib.auth.models import UserManager as BaseUserManager
from django.core.cache import caches
from django.db import transaction
from django.db.models import Q
from django.db.models.query import QuerySet

from events.accounts.utils import get_next_username
from events.common_app.directory import CachedDirectoryClient, build_real_user_name
from events.common_app.utils import get_user_ip_address


class UserQuerySet(QuerySet):
    def filter_uid_or_cloud_uid(self, uid, cloud_uid):
        conditions = Q()
        if uid:
            conditions |= Q(uid=uid)
        if cloud_uid:
            conditions |= Q(cloud_uid=cloud_uid)
        if not conditions:
            return self.none()
        return self.filter(conditions)


class UserManager(BaseUserManager):
    def get_queryset(self):
        return UserQuerySet(self.model, using=self._db)

    def filter_uid_or_cloud_uid(self, uid, cloud_uid):
        queryset = self.get_queryset()
        return queryset.filter_uid_or_cloud_uid(uid, cloud_uid)

    def _get_params_from_cloud(self, cloud_uid, dir_id):
        """
        Ходим в директорию, а не в паспорт, так как в паспорте
        нет этих пользователей
        """
        client = CachedDirectoryClient()
        fields = 'id,cloud_uid,name,email,nickname'
        userinfo = client.get_user(dir_id, uid=None, cloud_uid=cloud_uid, fields=fields)
        if userinfo:
            return {
                'default_email': userinfo.get('email') or userinfo.get('nickname'),
                'fields': {
                    'login': userinfo.get('nickname'),
                    'fio': build_real_user_name(userinfo.get('name'), None),
                },
            }

    def _get_params_from_cloud_cached(self, cloud_uid, dir_id):
        return self._get_params_from_cloud(cloud_uid, dir_id) or {}

    def get_params_from_cloud(self, cloud_uid, dir_id):
        return self._get_params_from_cloud_cached(cloud_uid, dir_id)

    def _get_params_from_passport(self, uid):
        bb = self._get_blackbox_instance()
        dbfields = [
            blackbox.FIELD_FIO,
            blackbox.FIELD_LOGIN,
            blackbox.FIELD_EMAIL,
        ]
        ip_address = get_user_ip_address()
        return bb.userinfo(uid, ip_address, emails='getdefault', by_login=False, dbfields=dbfields)

    def _get_params_from_passport_cached(self, uid):
        not_found = object()
        cache = caches['blackbox']
        cache_key = f'uid:{uid}'
        params = cache.get(cache_key, not_found)
        if params is not_found:
            params = self._get_params_from_passport(uid)
            cache.set(cache_key, params)
        return params

    def get_params_from_passport(self, uid):
        return self._get_params_from_passport_cached(uid)

    def _is_internal(self):
        # проверяем неоднозначность, чтобы в тестах корректно работали оверрайды
        return settings.IS_INTERNAL_SITE and not settings.IS_BUSINESS_SITE

    def _get_blackbox_instance(self):
        if self._is_internal():
            return settings.INTERNAL_SITE_BLACKBOX_INSTANCE
        else:
            return settings.EXTERNAL_SITE_BLACKBOX_INSTANCE

    def get_params(self, uid, cloud_uid, dir_id):
        params = None
        if cloud_uid:
            params = self._get_params_from_cloud_cached(cloud_uid, dir_id)
        if not params and uid:
            params = self._get_params_from_passport_cached(uid)
        return params or {}

    def get_or_create_user(self, uid, cloud_uid, dir_id, link_organizations=True):
        user = self.filter_uid_or_cloud_uid(uid, cloud_uid).first()
        if user:
            if (not user.uid and uid) or (not user.cloud_uid and cloud_uid):
                # подновляем uid или cloud_uid если необходимо
                user.uid = uid
                user.cloud_uid = cloud_uid
                user.save()
            if link_organizations:
                user.link_organizations_if_needed()
            return user

        userinfo = self.get_params(uid, cloud_uid, dir_id)
        return self.create_forms_user(
            uid=uid,
            cloud_uid=cloud_uid,
            email=userinfo.get('default_email'),
            username=userinfo.get('fields', {}).get('login'),
            link_organizations=link_organizations,
        )

    def create_forms_user(self, uid, cloud_uid, email, username, link_organizations=True):
        if not self._is_internal() or not username:
            username = get_next_username()

        with transaction.atomic():
            user = self.create_user(
                uid=uid,
                cloud_uid=cloud_uid,
                email=email,
                username=username,
            )
            if self._is_internal() and user.is_yandex_user():
                group = Group.objects.get(pk=settings.GROUP_ALLSTAFF_PK)
                user.groups.add(group)
            if link_organizations:
                user.link_organizations_if_needed()
        return user
