# -*- coding: utf-8 -*-
import logging

from datetime import timedelta
from django.contrib.auth.models import Group, AbstractUser
from django.conf import settings
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.db import models, transaction
from django.db.utils import DatabaseError
from django.utils import timezone
from django.utils.translation import ugettext as _

from events.common_app.middleware import get_current_request
from events.common_app.utils import generate_code
from events.accounts.managers import UserManager
from events.accounts.utils import (
    UID_TYPE_INTERNAL,
    UID_TYPE_EXTERNAL,
    UID_TYPE_CLOUD,
    is_yandex_user,
)
from events.common_app.jsonfield.fields import JSONField as JSONFieldWithFallback


logger = logging.getLogger(__name__)

CHOICES_UID_TYPE = {
    UID_TYPE_INTERNAL: UID_TYPE_INTERNAL,
    UID_TYPE_EXTERNAL: UID_TYPE_EXTERNAL,
    UID_TYPE_CLOUD: UID_TYPE_CLOUD,
}


class BaseKarmaError(Exception):
    def __init__(self, code, message):
        self.code = code
        super().__init__(message)


class UserKarmaError(BaseKarmaError):
    pass


class UserKarmaWarn(BaseKarmaError):
    pass


class Organization(models.Model):
    dir_id = models.CharField(max_length=16, null=True, unique=True)
    cloud_id = models.CharField(max_length=64, null=True, unique=True)
    created = models.DateTimeField(auto_now_add=True)
    modified = models.DateTimeField(auto_now=True)

    class Meta:
        db_table = 'dir_data_sync_organization'

    def __str__(self):
        if self.dir_id:
            return f'Organization(dir:{self.dir_id})'
        elif self.cloud_id:
            return f'Organization(cloud:{self.cloud_id})'
        return 'Organization()'


class User(AbstractUser):
    uid = models.CharField(max_length=255, blank=True, null=True, unique=True)
    cloud_uid = models.CharField(max_length=255, blank=True, null=True, unique=True)

    objects = UserManager()

    class Meta:
        db_table = 'auth_user'

    def get_params(self, dir_id=None):
        if settings.IS_TEST:
            from events.accounts.helpers import get_passport_data
            return get_passport_data(self.uid)
        if settings.IS_BUSINESS_SITE and not dir_id:
            request = get_current_request()
            if request and request.orgs:
                dir_id = request.orgs[0]
        return User.objects.get_params(self.uid, self.cloud_uid, dir_id)

    @property
    def params(self):
        params = getattr(self, '_params', None)
        if params is None:
            params = self.get_params()
            setattr(self, '_params', params)
        return params or {}

    @property
    def uid_type(self):
        if self.cloud_uid:
            return UID_TYPE_CLOUD
        elif self.uid:
            if settings.IS_INTERNAL_SITE:
                return UID_TYPE_INTERNAL
            else:
                return UID_TYPE_EXTERNAL

    def get_name_and_surname(self):
        return self.params.get('fields', {}).get('fio', '')

    def get_default_email(self):
        if self.uid_type == UID_TYPE_INTERNAL:
            return self.get_staff_email()
        return self.email or self.params.get('default_email', '')

    def get_name_and_surname_with_fallback(self):
        return self.get_name_and_surname() or self.email

    def follow(self, obj):
        from events.followme.models import ContentFollower

        follower, created = ContentFollower.objects.get_or_create(
            user=self,
            content_type=ContentType.objects.get_for_model(obj),
            object_id=obj.id,
            type='user',
            email=self.email,
        )
        if created:
            follower.secret_code = generate_code()
            follower.save()

        return follower, created

    def get_staff_email(self):
        return f'{self.username}@yandex-team.ru'

    def get_yandex_username(self):
        if self.uid_type == UID_TYPE_INTERNAL:
            return self.username
        return self.params.get('fields', {}).get('login', '')

    def get_karma(self):
        if not settings.IS_BUSINESS_SITE:
            return 0
        if self.is_anonymous or self.uid is None:
            return 0
        params = User.objects.get_params_from_passport(self.uid)
        karma = int(params.get('karma', '0'))
        logger.debug('User %s karma is %s', self.uid, karma)
        return karma

    def check_karma(self):
        from events.surveyme.models import Survey
        karma = self.get_karma()
        if karma > 75:
            logger.warn('User %s has bad karma %s', self.uid, karma)
        if karma > 85:
            raise UserKarmaError('bad-karma-85', _('Плохая карма, действия с формами запрещены'))
        if karma > 75:
            qs = (
                Survey.objects.using(settings.DATABASE_ROLOCAL)
                .filter(
                    user=self,
                    date_created__gt=timezone.now() - timedelta(hours=1),
                )
            )
            if qs.exists():
                raise UserKarmaError('bad-karma-75', _('Плохая карма, создавать новые формы запрещено чаще раза в час'))
            raise UserKarmaWarn('bad-karma-75', _('Плохая карма, публикация форм запрещена'))

    @property
    def is_anonymous(self):
        return not (self.uid or self.cloud_uid) or self.pk == settings.MOCK_PROFILE_ID

    def is_yandex_user(self):
        return is_yandex_user(self.uid)

    def _get_organizations(self, request=None):
        request = request or get_current_request()
        if not request:
            return []
        return request.orgs or []

    def link_organizations_if_needed(self):
        if not settings.IS_BUSINESS_SITE:
            return

        known_dirs = set(
            OrganizationToGroup.objects.filter(group__in=self.groups.all())
            .values_list('org__dir_id', flat=True)
        )
        requested_dirs = set(self._get_organizations())
        UserToGroup = User.groups.through

        to_insert = requested_dirs - known_dirs
        if to_insert:
            already_created = set(
                Organization.objects.filter(dir_id__in=to_insert)
                .values_list('dir_id', flat=True)
            )
            for dir_id in (to_insert - already_created):
                create_organization(dir_id)
            dbtype = (
                settings.DATABASE_DEFAULT
                if to_insert - already_created
                else settings.DATABASE_ROLOCAL
            )
            bulk = (
                UserToGroup(user=self, group=group)
                for group in (Group.objects.using(dbtype).filter(o2g__org__dir_id__in=to_insert))
            )
            UserToGroup.objects.bulk_create(bulk, ignore_conflicts=True)

        to_delete = known_dirs - requested_dirs
        if to_delete:
            UserToGroup.objects.filter(user=self, group__o2g__org__dir_id__in=to_delete).delete()

    def update_uids(self, uid, cloud_uid):
        if self.uid != uid or self.cloud_uid != cloud_uid:
            self.uid = uid
            self.cloud_uid = cloud_uid
            self.save(update_fields=['uid', 'cloud_uid'])

    def __str__(self):
        return str(self.pk)


class UserSettings(models.Model):
    user = models.OneToOneField(User, primary_key=True, on_delete=models.DO_NOTHING)
    settings = JSONFieldWithFallback(blank=True, null=True)


class OrganizationToGroup(models.Model):
    org = models.OneToOneField(Organization, primary_key=True, on_delete=models.DO_NOTHING)
    group = models.OneToOneField('auth.Group', null=True, on_delete=models.DO_NOTHING)

    class Meta:
        default_related_name = 'o2g'


def create_organization(dir_id):
    group_name = f'Все сотрудники {dir_id}'
    org = None

    try:
        with transaction.atomic():
            org = Organization.objects.create(dir_id=dir_id)
            group = Group.objects.create(name=group_name)
            OrganizationToGroup.objects.create(org=org, group=group)
    except DatabaseError:
        pass

    if not org:
        org = Organization.objects.using(settings.DATABASE_DEFAULT).get(dir_id=dir_id)
    return org


def get_or_create_organization(dir_id):
    try:
        org = Organization.objects.get(dir_id=dir_id)
    except Organization.DoesNotExist:
        org = None

    if not org:
        org = create_organization(dir_id)
    return org


class GrantTransitionInfo(models.Model):
    date_created = models.DateTimeField(auto_now_add=True)
    new_user = models.ForeignKey(User, related_name='+', on_delete=models.CASCADE)
    old_user = models.ForeignKey(User, related_name='+', on_delete=models.CASCADE)
    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_pk = models.CharField(max_length=255)
    content_object = GenericForeignKey('content_type', 'object_pk')

    class Meta:
        indexes = [
            models.Index(fields=['content_type', 'object_pk']),
        ]
