# -*- coding: utf-8 -*-
import os
import re

from django.conf import settings
from django.core.files.storage import Storage
from django.utils.deconstruct import deconstructible
from urllib.parse import urlparse

from events.avatars_extra.utils import generate_code, post_multipart
from events.common_app.utils import requests_session


class AvatarsStorageOld(Storage):
    read_methods = ['get']

    def __init__(self, host=None, host_for_write=None, namespace=None):
        self.host = host or settings.AVATARS_HOST
        self.host_for_write = host_for_write or settings.AVATARS_HOST_FOR_WRITE
        self.namespace = namespace or settings.AVATARS_NAMESPACE
        self.validate_params()
        self.clean_hostname = self.get_clean_hostname()

    def delete(self, name):
        delete_url = self._get_base_url_for_method(name=name, method='delete')
        response = requests_session.get(
            delete_url,
            timeout=settings.DEFAULT_TIMEOUT,
            verify=settings.YANDEX_ROOT_CERTIFICATE,
        )
        if response.status_code > 300:  # todo: test raise
            raise IOError('Avatars error. Could not delete image. Returned %s status for %s' %
                          (response.status_code, delete_url))

    def validate_params(self):
        if not self.host:
            raise AttributeError('Set AVATARS_NAMESPACE in project settings')
        elif not self.host_for_write:
            raise AttributeError('Set AVATARS_HOST_FOR_WRITE in project settings')
        elif not self.namespace:
            raise AttributeError('Set AVATARS_HOST in project settings')

    def get_clean_hostname(self):
        return urlparse(self.host).netloc

    def _clean_name(self, name):
        name = str(re.sub(r'[^\w]*', '', name))
        return name

    def _get_base_url_for_method(self, name, method='get'):
        host = self._get_host_for_method(method)
        method_and_namespace = '%s-%s' % (method, self.namespace)
        return os.path.join(host, method_and_namespace, name)

    def _get_host_for_method(self, method):
        if method in self.read_methods:
            return self.host
        else:
            return self.host_for_write

    def _save(self, name, content):
        data = {'id': name, 'file': content.file}

        url = self._get_base_url_for_method(name, 'put')
        post_multipart(url=url, data=data)

        return name

    def get_valid_name(self, name):
        # todo: test me
        """Подготавливает название файла к хранению в сторадже. (название файла без пути до него)

        Например:

            >> self.get_valid_name('файл.txt')
            file.txt

        В нашем случае название файла генерируется в случайный md5 hash

        """
        return generate_code()

    def get_available_name(self, name, max_length=255):
        # todo: test me
        """Возвращает имя, не занятое в сторедже

        Например:

            >> self.get_available_name('file.txt')
            _file.txt

        """
#        return 'after_available_%s' % name
        return self._clean_name(name)

    def url(self, name):
        return self._get_thumb_url(name=name)

    def _get_thumb_url(self, name, width=None, height=None, alias='orig'):
        # tested from AvatarFieldFile
        if all(param is not None for param in (width, height)):
            size_or_alias = '%sx%s' % (width, height)
        else:
            size_or_alias = alias
        return os.path.join(self._get_base_url_for_method(name=name, method='get'), size_or_alias)


@deconstructible
class AvatarsStorage(AvatarsStorageOld):
    def __init__(self, host=None, host_for_write=None, namespace=None):
        self.host = host or settings.AVATARS_HOST
        self.host_for_write = host_for_write or settings.AVATARS_HOST_FOR_WRITE
        self.namespace = namespace or settings.AVATARS_NAMESPACE
        super(AvatarsStorage, self).__init__(self.host, self.host_for_write, self.namespace)

    def _save(self, name, content):
        files = {'file': (name, content.file)}
        url = self._get_base_url_for_method(name, 'put')
        response = requests_session.post(
            url,
            files=files,
            timeout=settings.AVATARS_TIMEOUT,
            verify=settings.YANDEX_ROOT_CERTIFICATE,
        )
        response.raise_for_status()
        response_json = response.json()
        return '%s/%s' % (response_json['group-id'], name)
