# -*- coding: utf-8 -*-
import time
import socket
import xmlrpc.client
import logging

from django.conf import settings
from django.utils.encoding import force_str


logger = logging.getLogger(__name__)


class BalanceClient(object):
    def __init__(self, token):
        self.token = token
        self.server = xmlrpc.client.ServerProxy(settings.BALANCE_API_URL)

    def create_service_product(self, service_product_id, name, currency, product_qty=None, prices=None):
        params = {
            'service_product_id': str(service_product_id),
            'name': name,
        }
        if product_qty:
            params['product_qty'] = product_qty
        if prices:
            params['prices'] = [
                {
                    'price': price,
                    'region_id': 225,  # Россия из геобазы
                    'dt': int(time.time()),  # с какого времени действительна цена
                    'currency': currency,
                } for price in prices
            ]
        response = self._make_request('CreateServiceProduct', params)
        logger.info(
            'Creating service product with id=%s.\nRequest: %s\nResponse: %s',
            service_product_id,
            params,
            response,
        )
        return response

    def create_order_or_subscription(self, service_product_id, uid, user_ip, qty=1, developer_payload=None):
        params = {
            'service_product_id': str(service_product_id),
            'region_id': 225,
            'qty': qty,
        }
        if uid:
            params['uid'] = str(uid)
        if developer_payload:
            params['developer_payload'] = developer_payload
        response = self._make_request('CreateOrderOrSubscription', params)
        logger.info(
            'Creating order.\nRequest: %s\nResponse: %s',
            params,
            response,
        )
        return response

    def create_basket(self, orders, uid, user_ip, currency='RUB', payment_timeout=60*15, return_path=None):
        params = {
            'payment_timeout': payment_timeout,
            'orders': orders,
            'currency': currency,
        }
        if uid:
            params['uid'] = str(uid)
        if return_path:
            params['return_path'] = return_path
        response = self._make_request('CreateBasket', params)
        logger.info(
            'Creating basket.\nRequest: %s\nResponse: %s',
            params,
            response,
        )
        return response

    def pay_basket(self, uid, user_ip, trust_payment_id):
        params = {
            'user_ip': user_ip,
            'trust_payment_id': trust_payment_id,
        }
        if uid:
            params['uid'] = str(uid)
        response = self._make_request('PayBasket', params)
        logger.info(
            'Start payment process for basket with trust_payment_id=%s.\nRequest: %s\nResponse: %s',
            trust_payment_id,
            params,
            response,
        )
        return response

    def check_basket(self, uid, user_ip, trust_payment_id):
        params = {
            'user_ip': user_ip,
            'trust_payment_id': trust_payment_id,
        }
        if uid:
            params['uid'] = str(uid)
        response = self._make_request('CheckBasket', params)
        logger.info(
            'Check basket with trust_payment_id=%s.\nRequest: %s\nResponse: %s',
            trust_payment_id,
            params,
            response,
        )
        return response

    def get_service_product(self, service_product_id):
        params = {
            'service_product_id': str(service_product_id),
        }
        response = self._make_request('GetServiceProduct', params)
        logger.info(
            'Get service product with id=%s.\nRequest: %s\nResponse: %s',
            service_product_id,
            params,
            response,
        )
        return response

    def _make_request(self, method_name, params, try_count=1):
        try:
            return getattr(self.server, method_name)(self.token, params)
        except socket.error as e:
            logger.error(
                'Error connecting to balance. Method:%s\nRequest: %s\nError:%s\ntry_count: %s',
                method_name,
                params,
                force_str(e),
                try_count,
            )
            if try_count <= settings.BALANCE_CLIENT_RETRY_LIMIT_COUNT:
                time.sleep(0.5)
                return self._make_request(method_name, params, try_count=try_count + 1)
            else:
                raise e
