# -*- coding: utf-8 -*-
import json
from dateutil import parser
import logging

from django.conf import settings
from django.db import models
from django.utils.encoding import force_str

from events.surveyme.models import Survey, ProfileSurveyAnswer
from events.balance import exceptions as balance_exceptions
from events.balance import utils as balance_utils


logger = logging.getLogger(__name__)


class Ticket(models.Model):
    survey = models.OneToOneField(Survey, related_name='tickets_info', on_delete=models.DO_NOTHING)
    price = models.IntegerField()
    currency = models.CharField(max_length=3, default='RUB')
    name = models.CharField(max_length=255)
    quantity = models.PositiveIntegerField(null=True, blank=True)
    trust_return_path = models.CharField(blank=True, max_length=255)
    trust_unsuccessful_return_path = models.CharField(blank=True, max_length=255)
    is_can_pay_by_yandex_money = models.BooleanField(default=False)

    def buy_tickets(self, profile_survey_answer, user_ip, quantities, payment_timeout=None, return_path=None, start_payment=True):
        if payment_timeout is None:
            payment_timeout = settings.BALANCE_PAYMENT_TIMEOUT
        return_path = self.trust_return_path or return_path
        developer_payload = self._get_developer_payload_for_orders(quantities)
        order = Order.create(
            ticket=self,
            profile_survey_answer=profile_survey_answer,
            quantities=quantities,
            user_ip=user_ip,
            payment_timeout=payment_timeout,
            return_path=return_path,
            developer_payload=developer_payload,
        )
        trust_form_data = None
        if start_payment:
            trust_form_data = order.start_payment_process()
            if self.is_can_pay_by_yandex_money:
                payment_form = trust_form_data['payment_form']
                payment_form['scid'] = settings.YANDEX_MONEY_SCID
                payment_form['response_type'] = settings.YANDEX_MONEY_RESPONSE_TYPE
        return order, trust_form_data

    def get_status(self):
        balance_fields_to_show = {
            'available_qty': 'available_quantity',
        }
        balance_client = balance_utils.get_balance_client_for_currency(self.currency)
        service_product_info = balance_client.get_service_product(self.id)
        data = {v: service_product_info.get(k) for k, v in balance_fields_to_show.items()}
        data.update({
            'price': self.price,
            'currency': self.currency,
            'quantity_is_limited': 'available_qty' in service_product_info
        })
        return data

    def _get_developer_payload_for_orders(self, quantities):
        developer_payload_data = {}
        for text in self.survey.texts.filter(slug__in=['payment_page_title', 'successful_payment']):
            developer_payload_data[text.slug] = text.value_ru
        developer_payload_data['is_can_pay_by_yandex_money'] = self.is_can_pay_by_yandex_money
        developer_payload_data['quantity_of_tickets'] = sum(map(int, quantities.values()))
        developer_payload_data['unsuccessful_return_path'] = self.trust_unsuccessful_return_path
        return json.dumps(developer_payload_data)


class Order(models.Model):
    profile_survey_answer = models.ForeignKey(ProfileSurveyAnswer, related_name='orders', on_delete=models.DO_NOTHING)
    ticket_info = models.ForeignKey(Ticket, related_name='orders', on_delete=models.DO_NOTHING)
    trust_payment_id = models.CharField(max_length=255)
    metadata = models.TextField(blank=True)
    payment_status = models.CharField(max_length=255, blank=True)
    payment_status_code = models.CharField(max_length=255, blank=True)
    payment_start_datetime = models.DateTimeField(null=True)
    payment_timeout = models.SmallIntegerField(null=True)
    user_ip = models.GenericIPAddressField()

    def __str__(self):
        return 'Заказ %s. Статус: %s' % (self.pk, self._get_readable_status())

    def _get_readable_status(self):
        statuses = {
            'success': 'Оплачен',
            'error': 'Ошибка',
            'no_auth': 'no_auth',
            'cancelled': 'Отменён',
            'wait_for_notification': 'В ожидании оплаты'
        }
        return force_str(statuses.get(self.payment_status, self.payment_status))

    class Meta:
        verbose_name = 'Заказ'
        verbose_name_plural = 'Заказы'

    @classmethod
    def create(cls, ticket, profile_survey_answer, quantities, user_ip, payment_timeout=None, return_path=None, developer_payload=None):
        uid = cls.get_uid_from_profile_survey_answer(profile_survey_answer)
        if payment_timeout is None:
            payment_timeout = settings.BALANCE_PAYMENT_TIMEOUT
        balance_orders = [
            BalanceOrder.create(uid, user_ip, ticket, price, qty, developer_payload) for price, qty in quantities.items()
        ]
        trust_payment_id = cls._create_basket_in_balance(
            ticket,
            uid,
            user_ip,
            balance_orders,
            payment_timeout,
            return_path,
        )
        order = cls.objects.create(
            profile_survey_answer=profile_survey_answer,
            user_ip=user_ip,
            trust_payment_id=trust_payment_id,
            ticket_info=ticket,
        )
        order.balance_orders.add(*balance_orders)
        return order

    @staticmethod
    def get_uid_from_profile_survey_answer(profile_survey_answer):
        user = profile_survey_answer.user
        return user.uid

    @classmethod
    def _create_basket_in_balance(cls, ticket, uid, user_ip, balance_orders, payment_timeout, return_path):
        orders = [
            {
                'service_order_id': order.service_order_id,
                'qty': order.quantity,
                'fiscal_nds': 'nds_18',
                'fiscal_title': ticket.name,
            }
            for order in balance_orders
        ]
        balance_client = balance_utils.get_balance_client_for_currency(ticket.currency)
        result = balance_client.create_basket(
            uid=uid,
            user_ip=user_ip,
            orders=orders,
            currency=ticket.currency,
            payment_timeout=payment_timeout,
            return_path=return_path,
        )
        if result.get('status') == 'success':
            return result['trust_payment_id']
        elif result.get('status') == 'error' and result.get('status_code') == 'service_product_ended':
            logger.error('Basket creating error! %s', result)
            raise balance_exceptions.BalanceServiceProductEnded('The tickets ran out. Survey: %s' % ticket.survey_id)
        else:
            logger.error('Basket creating error! %s', result)
            raise balance_exceptions.BalanceBasketCreatingError('Basket creating error! %s' % result)

    def check_status(self):
        balance_client = balance_utils.get_balance_client_for_currency(self.ticket_info.currency)
        result = balance_client.check_basket(
            user_ip=self.user_ip,
            uid=self.get_uid_from_profile_survey_answer(self.profile_survey_answer),
            trust_payment_id=self.trust_payment_id,
        )
        if self.payment_status != result.get('status'):
            self.payment_status = result.get('status')
            self.payment_status_code = result.get('status_code', '')
            self.save(update_fields=['payment_status', 'payment_status_code'])
            return self.payment_status

    def start_payment_process(self):
        balance_client = balance_utils.get_balance_client_for_currency(self.ticket_info.currency)
        result = balance_client.pay_basket(
            user_ip=self.user_ip,
            uid=self.get_uid_from_profile_survey_answer(self.profile_survey_answer),
            trust_payment_id=self.trust_payment_id,
        )
        self.payment_status = result.get('status')
        if result.get('start_ts'):
            self.payment_start_datetime = parser.parse(result.get('start_ts'))
        self.save(update_fields=['payment_status', 'payment_start_datetime'])
        return result


class BalanceOrder(models.Model):
    order = models.ForeignKey(Order, blank=True, null=True, related_name='balance_orders', on_delete=models.DO_NOTHING)
    service_order_id = models.CharField(max_length=255)
    price = models.PositiveIntegerField()
    quantity = models.PositiveSmallIntegerField(default=1)
    currency = models.CharField(max_length=3)

    @classmethod
    def create(cls, uid, user_ip, ticket, price, qty=1, developer_payload=None):
        service_order_id = cls._create_order_in_balance(uid, user_ip, ticket, qty, developer_payload)
        return cls.objects.create(service_order_id=service_order_id, quantity=qty, price=price, currency=ticket.currency)

    @classmethod
    def _create_order_in_balance(cls, uid, user_ip, ticket, qty=1, developer_payload=None):
        balance_client = balance_utils.get_balance_client_for_currency(ticket.currency)
        result = balance_client.create_order_or_subscription(
            uid=uid,
            user_ip=user_ip,
            service_product_id=ticket.pk,
            qty=qty,
            developer_payload=developer_payload,
        )
        if result.get('status') == 'success':
            return result['service_order_id']
        elif result.get('status') == 'error' and result.get('status_code') == 'service_product_ended':
            logger.error('Order creating error! %s', result)
            raise balance_exceptions.BalanceServiceProductEnded('Нет запрошенного количества билетов.')
        else:
            logger.error('Order creating error! %s', result)
            raise balance_exceptions.BalanceOrderCreatingError('Order creating error! %s' % result)
