# -*- coding: utf-8 -*-
from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _
from django.conf import settings

from events.captcha.captcha import Captcha


class CaptchaKeyHiddenWidget(forms.HiddenInput):
    def render(self, name, value, attrs=None):
        captcha_type = self.attrs['captcha_type']
        image_url, captcha_key = Captcha().generate(type=captcha_type)

        value = captcha_key
        if attrs is None:
            attrs = {}
        attrs['data-captcha-image-url'] = image_url
        return super(CaptchaKeyHiddenWidget, self).render(name, value, attrs)


class CaptchaTextInput(forms.MultiWidget):
    def __init__(self, attrs):
        widgets = (
            CaptchaKeyHiddenWidget(attrs),  # captcha key
            forms.TextInput(attrs),  # user input
        )
        super(CaptchaTextInput, self).__init__(widgets=widgets, attrs=attrs)

    def decompress(self, value):
        """
        The rationale behind “decompression” is that it is necessary to “split” the combined value of the form
        field into the values for each widget.
        """
        if value:
            return value['captcha_key'], value['user_input']
        else:
            return [None, None]


class CaptchaField(forms.MultiValueField):
    widget = CaptchaTextInput
    default_error_messages = {
        'invalid': _('enter_a_valid_value'),
    }

    def __init__(self, fields=(), *args, **kwargs):
        fields = (
            forms.CharField(),  # captcha key
            forms.CharField(),  # user input
        )
        self.captcha_type = kwargs.pop('captcha_type', settings.CAPTCHA_TYPE)
        kwargs['widget'] = self.widget(attrs={'captcha_type': self.captcha_type})
        super(CaptchaField, self).__init__(fields, *args, **kwargs)

    def compress(self, data_list):
        """
        Takes a list of valid values and returns a “compressed” version of those values – in a single value.
        For example, SplitDateTimeField is a subclass which combines a time field and a date
        field into a datetime object.
        """
        if data_list:
            return {
                'captcha_key': data_list[0],
                'user_input': data_list[1],
            }
        else:
            return None

    def validate(self, value):
        is_checked, error_code = Captcha().check(
            type=self.captcha_type,
            rep=value['user_input'],
            key=value['captcha_key'],
        )
        if not is_checked:
            raise ValidationError(self.error_messages['invalid'])
