# -*- coding: utf-8 -*-
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db.models import Q
from django.http import Http404
from django.utils.timezone import now
from django.utils.translation import get_language

from rest_framework.response import Response
from rest_framework import status, views, viewsets

from django_filters.filterset import FilterSet
from django_filters.filters import CharFilter, BaseInFilter

from events.common_app.api_admin.v2.serializers import (
    ContentTypeSerializer,
    DirectorySuggestSerializer,
    UserMessageSerializer,
)
from events.common_app.directory import DirectorySuggest, BiSearchSuggest
from events.rest_framework_contrib.mixins import (
    InternalGenericApiViewV2Mixin,
)
from events.common_app.models import UserMessage
from events.celery_app import app


class CharInFilter(BaseInFilter, CharFilter):
    pass


class ContentTypeFilter(FilterSet):
    model__in = CharInFilter(field_name='model', lookup_expr='in')

    class Meta:
        model = ContentType
        fields = (
            'app_label',
            'model',
        )


class ContentTypeViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = ContentTypeSerializer
    queryset = ContentType.objects.all()
    filter_class = ContentTypeFilter
    # todo: permissions


class TaskViewSet(viewsets.ViewSet):
    def retrieve(self, request, *args, **kwargs):
        task = self.get_task_or_404()
        data = {
            'meta': {
                'failed': task.failed(),
                'ready': task.ready(),
                'progress': {
                    'do': self.get_task_info(task),
                    'finished': task.ready(),
                },
            },
            'result': None,
            'finished': task.ready(),
            'failed': task.failed(),
            'message': self.get_task_info(task),
        }
        if task.ready() and not task.failed():
            data['result'] = task.result
        return Response(data)

    def get_task_info(self, task):
        if task.ready():
            if task.failed():
                return str(task.info)
            else:
                return 'Работа закончена'
        return 'Работа продолжается'

    def destroy(self, request, *args, **kwargs):
        task = self.get_task_or_404()
        task.revoke(terminate=True)
        return Response(None, status=status.HTTP_204_NO_CONTENT)

    def get_task_or_404(self):
        task = self.get_task()
        if not task:
            raise Http404
        return task

    def get_task(self):
        task_id = self.kwargs.get('pk')
        if task_id:
            return app.AsyncResult(task_id)


class LanguageViewSet(viewsets.ViewSet):
    def get_languages(self, request):
        return {
            _id: name
            for _id, name in settings.LANGUAGES
        }

    def list(self, request, *args, **kwargs):
        languages = self.get_languages(request)
        return Response([
            {
                'id': _id,
                'name': name,
            }
            for _id, name in languages.items()
        ])

    def retrieve(self, request, *args, **kwargs):
        languages = self.get_languages(request)
        _id = kwargs.get('pk')
        name = languages.get(_id)
        if not name:
            raise Http404
        return Response({
            'id': _id,
            'name': name,
        })


class UserMessageViewSet(InternalGenericApiViewV2Mixin, viewsets.ReadOnlyModelViewSet):
    serializer_class = UserMessageSerializer
    queryset = UserMessage.objects.filter(enabled=True)

    def get_queryset(self, *args, **kwargs):
        queryset = super().get_queryset(*args, **kwargs)
        queryset = queryset.filter(Q(date_started__isnull=True) | Q(date_started__lte=now()))
        queryset = queryset.filter(Q(date_finished__isnull=True) | Q(date_finished__gte=now()))
        return queryset


class DirectorySuggestView(InternalGenericApiViewV2Mixin, views.APIView):
    def get(self, request, *args, **kwargs):
        serializer = DirectorySuggestSerializer(data=request.query_params)
        serializer.is_valid(raise_exception=True)

        dir_id = serializer.validated_data['org_id']
        if dir_id not in request.orgs:
            return Response({})

        if settings.USE_BISEARCH:
            ds = BiSearchSuggest(request.user.uid, request.user.cloud_uid, dir_id, get_language())
        else:
            ds = DirectorySuggest(dir_id, get_language())
        data = ds.suggest(
            text=serializer.validated_data.get('text'),
            ids=serializer.validated_data.get('id'),
            layers=serializer.validated_data.get('layers'),
        )
        return Response(data)
