# -*- coding: utf-8 -*-
import json
import hashlib
import logging

from blackbox import (
    Blackbox as BaseBlackbox,
    BlackboxError,
    TVM2_RETRY_ATTEMPT,
    TVM2_SERVICE_TICKET_HEADER,
)
from io import BytesIO
from requests.exceptions import ConnectionError, HTTPError

from events.common_app.utils import requests_session

logger = logging.getLogger(__name__)

HTTP_HEADERS = {}


class Blackbox(BaseBlackbox):
    def _blackbox_http_call(self, params, headers=None, timeout=None):
        default_headers = HTTP_HEADERS.copy()

        if self.use_tvm2:
            tvm2_client = self._get_tvm2_client()
            for _ in range(TVM2_RETRY_ATTEMPT):
                logger.info('Getting tvm2 ticket for blackbox')
                service_tickets = tvm2_client.get_service_tickets(
                    self.blackbox_client_id
                )
                blackbox_ticket = service_tickets.get(self.blackbox_client_id)
                if blackbox_ticket:
                    logger.info('Successfully get tvm2 ticket for blackbox')
                    default_headers[TVM2_SERVICE_TICKET_HEADER] = blackbox_ticket
                    break
            else:
                logger.error('Couldn\'t get service ticket for blackbox')

        if params.get('method', '') == 'login':
            httpmethod = 'POST'
            default_headers['Content-Type'] = 'application/x-www-form-urlencoded'
        else:
            httpmethod = 'GET'

        default_headers.update(headers or {})

        try:
            resp = requests_session.request(
                method=httpmethod,
                url=self.url,
                params=params,
                headers=default_headers,
                timeout=timeout
            )
        except ConnectionError as ex:
            raise BlackboxError('ошибка сокета: %s' % ex)
        except HTTPError as ex:
            raise BlackboxError('ошибка HTTP: %s' % ex)

        # По документации ЧЯ всегда возвращает 200, поэтому любой другой код
        # считается ошибкой.
        if resp.status_code != 200:
            raise BlackboxError('получен код состояния %d, ожидался 200' %
                                resp.status_code)

        content_type = resp.headers.get('content-type', 'text/xml').split(';')[0]

        format = params.get('format', 'xml')

        if format == 'json' and content_type != 'application/json':
            raise BlackboxError('получен %s, ожидался JSON' % content_type)
        elif format == 'xml' and content_type not in ('text/xml', 'application/xml'):
            raise BlackboxError('получен %s, ожидался XML' % content_type)

        return resp.content


class JsonBlackbox(Blackbox):
    def __init__(self, *args, **kwargs):
        super(JsonBlackbox, self).__init__(*args, **kwargs)
        self._json_decode = kwargs.get('json_decode_function')

    def _get_json_decode(self):
        return json.loads

    def json_decode(self, data):
        if not self._json_decode:
            self._json_decode = self._get_json_decode()
        return self._json_decode(data.decode())

    def _blackbox_json_call(self, method, **kwargs):
        response = self._blackbox_call(method, format='json', **kwargs)
        return self.json_decode(response)

    def userinfo(self, **kwargs):
        if 'uid' in kwargs:
            uid = kwargs['uid']
            if not isinstance(uid, (int, str)):
                kwargs['uid'] = ','.join(str(u) for u in uid)
        return self._blackbox_json_call('userinfo', **kwargs)


class CachedBlackbox(Blackbox):
    SALT = b'\xe6\xe7\xb8!vc'

    def get_cache(self):
        from django.core.cache import caches
        return caches['blackbox']

    def get_cache_key(self, *args):
        buf = BytesIO()
        for arg in args:
            if isinstance(arg, bytes):
                buf.write(arg)
            elif isinstance(arg, str):
                buf.write(arg.encode('utf-8'))
            else:
                buf.write(str(arg).encode('utf-8'))
        buf.write(self.SALT)
        return hashlib.md5(buf.getvalue()).hexdigest()

    def sessionid(self, session_id, *args, **kwargs):
        cache = self.get_cache()
        session_key = self.get_cache_key(session_id)
        session = cache.get(session_key)
        if session is None:
            session = super(CachedBlackbox, self).sessionid(session_id, *args, **kwargs)
            cache.set(session_key, session)
        return session

    def userinfo(self, login_or_uid, userip, *args, **kwargs):
        cache = self.get_cache()
        userinfo_key = self.get_cache_key(login_or_uid, userip)
        userinfo = cache.get(userinfo_key)
        if userinfo is None:
            userinfo = super(CachedBlackbox, self).userinfo(login_or_uid, userip, *args, **kwargs)
            cache.set(userinfo_key, userinfo)
        return userinfo
