# -*- coding: utf-8 -*-
import re
from io import StringIO
from django.conf import settings
from events.common_app.utils import requests_session, chunks
from events.staff.models import StaffPerson


def get_dataset_info(dataset_key):
    headers = {
        'Authorization': 'OAuth %s' % settings.DATALENS_TOKEN,
    }
    url = settings.DATALENS_URL.format(
        host=settings.DATALENS_HOST,
        dataset_key=dataset_key,
    )
    response = requests_session.get(
        url,
        headers=headers,
        timeout=settings.DEFAULT_TIMEOUT,
        verify=settings.YANDEX_ROOT_CERTIFICATE,
    )
    response.raise_for_status()
    return response.json()


def put_dataset_info(dataset_key, dataset_info):
    headers = {
        'Authorization': 'OAuth %s' % settings.DATALENS_TOKEN,
    }
    url = settings.DATALENS_URL.format(
        host=settings.DATALENS_HOST,
        dataset_key=dataset_key,
    )
    response = requests_session.put(
        url,
        json=dataset_info,
        headers=headers,
        timeout=(1, 45),
        verify=settings.YANDEX_ROOT_CERTIFICATE,
    )
    response.raise_for_status()


def new_logins(rls_data, logins):
    if not logins:
        return

    new_logins = set(check_logins(logins))
    if not new_logins:
        return

    old_logins = set(get_rls_logins(rls_data))
    if old_logins.issuperset(new_logins):
        return

    new_logins.update(old_logins)

    st = StringIO()
    first = True
    for chunk in chunks(new_logins, 100):
        for login in check_logins(chunk):
            if not first:
                st.write('\n')
            st.write("'{login}': {login}".format(login=login))
            first = False
    return st.getvalue()


def get_rls_logins(rls_data):
    login_re = re.compile(r"'([^']+)':\s+\1")
    for row in rls_data.split('\n'):
        m = login_re.match(row)
        if m and m.group(1):
            yield m.group(1)


def check_logins(logins):
    return (
        StaffPerson.objects.using(settings.DATABASE_ROLOCAL)
        .filter(login__in=logins, is_dismissed=False)
        .values_list('login', flat=True)
    )


def get_column_guid(dataset_info, source):
    result_schema = dataset_info['result_schema']
    for column in result_schema:
        if column['source'] == source:
            return column['guid']


def get_rls_for_column(dataset_info, column_guid):
    rls_data = dataset_info['rls']
    return rls_data.get(column_guid)


def set_rls_for_column(dataset_info, column_guid, rls_data):
    dataset_info['rls'][column_guid] = rls_data


class DataLensError(Exception):
    pass


class DataLensColumnNotFound(DataLensError):
    pass


class DataLensRlsDataNotFound(DataLensError):
    pass


class DataLensNothingToAppend(DataLensError):
    pass


def add_new_logins(dataset_key, logins):
    dataset_info = get_dataset_info(dataset_key)

    login_guid = get_column_guid(dataset_info, 'login')
    if not login_guid:
        raise DataLensColumnNotFound

    rls_data = get_rls_for_column(dataset_info, login_guid)
    if rls_data is None:
        raise DataLensRlsDataNotFound

    changed_rls_data = new_logins(rls_data, logins)
    if not changed_rls_data:
        raise DataLensNothingToAppend

    set_rls_for_column(dataset_info, login_guid, changed_rls_data)

    put_dataset_info(dataset_key, dataset_info)
