# -*- coding: utf-8 -*-
import logging
import re
import os.path

from django.conf import settings
from django.utils.translation import ugettext as _
from requests import Session
from requests.adapters import HTTPAdapter
from requests.exceptions import RequestException
from requests.packages.urllib3.util.retry import Retry
from urllib.parse import urlencode, urlunsplit

from events.common_app.utils import requests_session
from events.yauth_contrib.auth import TvmAuth

logger = logging.getLogger(__name__)


class DiskUploadError(Exception):
    def __init__(self, status_code, message):
        self.status_code = status_code
        self.message = message


class DiskClient:
    max_upload_retries = 3

    def __init__(self, user_uid):
        self.user_uid = user_uid
        self.auth = TvmAuth(settings.DISK_TVM2_CLIENT)
        self.upload_content_session = self._get_upload_content_session()

    def _make_request(self, method, url_path, params, data=None):
        url = urlunsplit(('https', settings.DISK_HOST, url_path, urlencode(params), ''))
        try:
            return requests_session.request(
                method,
                url=url,
                headers={
                    'X-Uid': str(self.user_uid),
                },
                json=data,
                auth=self.auth,
                verify=settings.YANDEX_ROOT_CERTIFICATE,
                timeout=settings.DISK_TIMEOUT,
            )
        except RequestException as e:
            logger.error('DiskError: Requests error, method=%s, url=%s, exception=%s', method, url, e)
            raise

    def _get_upload_content_session(self):
        max_retries = Retry(
            self.max_upload_retries,
            method_whitelist=('POST', 'PUT'),
            status_forcelist=(500, 502, 503, 504),
            raise_on_status=False,
            backoff_factor=0.2,
        )
        adapter = HTTPAdapter(max_retries=max_retries)
        session = Session()
        session.mount('https://', adapter)
        return session

    def _upload_content(self, method, url, content):
        try:
            return self.upload_content_session.request(
                method,
                url=url,
                data=content,
                verify=settings.YANDEX_ROOT_CERTIFICATE,
                timeout=settings.DISK_TIMEOUT,
            )
        except RequestException as e:
            logger.error('DiskError: Requests error, method=%s, url=%s, exception=%s', method, url, e)
            raise

    def _get_params(self, **kwargs):
        return {
            key: value
            for key, value in kwargs.items()
            if value is not None
        }

    def get_resource(self, path, fields=None, limit=None, offset=None):
        url_path = '/v1/disk/resources'
        params = self._get_params(path=path, fields=fields, limit=limit, offset=offset)
        return self._make_request('get', url_path, params)

    def put_resource(self, path, fields=None):
        url_path = '/v1/disk/resources'
        params = self._get_params(path=path, fields=fields)
        return self._make_request('put', url_path, params)

    def delete_resource(self, path, fields=None, force_async=None, permanently=None):
        url_path = '/v1/disk/resources'
        params = self._get_params(path=path, fields=fields, force_async=force_async, permanently=permanently)
        return self._make_request('delete', url_path, params)

    def patch_resource(self, path, data, fields=None):
        url_path = '/v1/disk/resources'
        params = self._get_params(path=path, fields=fields)
        return self._make_request('patch', url_path, params, data=data)

    def get_resource_upload(self, path, fields=None, overwrite=None):
        url_path = '/v1/disk/resources/upload'
        params = self._get_params(path=path, fields=fields, overwrite=overwrite)
        return self._make_request('get', url_path, params)

    def get_resource_online_editor(self, path, fields=None):
        url_path = '/v1/disk/resources/online-editor'
        params = self._get_params(path=path, fields=fields)
        return self._make_request('get', url_path, params)

    def get_resource_download(self, path, fields=None):
        url_path = '/v1/disk/resources/download'
        params = self._get_params(path=path, fields=fields)
        return self._make_request('get', url_path, params)

    def isfile(self, path):
        response = self.get_resource(path, fields='type')
        if response.status_code != 200:
            return False
        data = response.json()
        return data['type'] == 'file'

    def isdir(self, path):
        response = self.get_resource(path, fields='type')
        if response.status_code != 200:
            return False
        data = response.json()
        return data['type'] == 'dir'

    def mkdir(self, path):
        if path == '/' or path == 'disk:/':
            return
        if not self.isdir(path):
            parent, _ = os.path.split(path)
            self.mkdir(parent)
            self.put_resource(path)

    def upload(self, path, content, overwrite=True):
        response = self.get_resource_upload(path, overwrite=overwrite)
        data = response.json()
        if response.status_code != 200:
            logger.error(
                'DiskError: Can\'t get upload url, status=%s, response=%s',
                response.status_code,
                response.text,
            )
            raise DiskUploadError(response.status_code, data.get('message'))

        response = self._upload_content(data['method'].lower(), data['href'], content)
        if response.status_code not in (200, 201, 202):
            msg = _('Ошибка записи файла на Яндекс.Диск')
            if response.status_code == 503:
                msg = _('Сервис временно недоступен')
            elif response.status_code == 507:
                msg = _('Недостаточно свободного места')
            logger.error(
                'DiskError: Can\'t upload file, status=%s, response=%s',
                response.status_code,
                response.text,
            )
            raise DiskUploadError(response.status_code, msg)

    def set_metadata(self, path, metadata):
        data = {
            'custom_properties': metadata,
        }
        response = self.patch_resource(path, data, fields='custom_properties')
        return response.status_code == 200

    def get_metadata(self, path):
        response = self.get_resource(path, fields='custom_properties')
        if response.status_code == 200:
            data = response.json()
            return data['custom_properties']

    def get_interface_url(self, path):
        return 'https://%s/client/disk%s' % (settings.DISK_FRONTEND_HOST, path)

    def get_download_url(self, path):
        response = self.get_resource_download(path)
        if response.status_code == 200:
            data = response.json()
            return data['href']

    def get_edit_url(self, path):
        response = self.get_resource_online_editor(path)
        if response.status_code == 200:
            data = response.json()
            return data['edit_url']

    def listdir(self, path):
        offset = 0
        limit = 100
        fields = 'type,_embedded'
        while True:
            response = self.get_resource(path, fields=fields, offset=offset, limit=limit)
            if response.status_code != 200:
                return
            data = response.json()
            embedded = data['_embedded']
            for item in embedded['items']:
                yield item['name']
            if offset + limit > embedded['total']:
                break
            offset += limit

    def check_file_name(self, file_path):
        """
        Здесь ищем наиболее подходящее имя для файла,
        так чтобы не перезаписать новым контентом старые данные.
        Если файл с нужным нам именем существует, перебираем
        все файлы в каталоге и ищем похожие.
        Пример: нам нужно записать файл с именем `newbook.xlsx`, в
        каталоге уже есть файлы `newbook.xlsx`, `newbook 2.xlsx`,
        тогда функция вернет `newbook 3.xlsx`
        """
        if not self.isfile(file_path):
            return file_path, False

        digits_re = re.compile(r' (\d+)')
        folder_path = os.path.dirname(file_path)
        file_name = os.path.basename(file_path)
        file_part, ext_part = os.path.splitext(file_name)
        max_index = 1
        for name in self.listdir(folder_path):
            if name.startswith(file_part) and name.endswith(ext_part):
                name_part, _ = os.path.splitext(name)
                suffix = name_part[len(file_part):]
                m = digits_re.search(suffix)
                if m:
                    max_index = max(max_index, int(m.group(1)))
        max_index += 1
        file_name = '%s %s%s' % (file_part, max_index, ext_part)
        return os.path.join(folder_path, file_name), True
