# -*- coding: utf-8 -*-
import json

from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext as _
from django.core.validators import EMPTY_VALUES
from django.utils.datastructures import MultiValueDict

from events.common_app.utils import retry
from events.common_storages.models import ProxyStorageModel
from events.common_app.widgets import NumberInput


class IntegerField(forms.IntegerField):
    widget = NumberInput


class SelectWidget(forms.Select):
    def value_from_datadict(self, data, files, name):
        if isinstance(data, MultiValueDict):
            return data.getlist(name)
        return data.get(name, None)


class MultiFileInput(forms.FileInput):
    def render(self, name, value, attrs=None, renderer=None):
        attrs['multiple'] = 'true'
        return super(MultiFileInput, self).render(name, value, attrs, renderer)

    def value_from_datadict(self, data, files, name):
        if hasattr(data, 'getlist'):
            return data.getlist(name)
        else:
            return [data.get(name)]


class MultiFileField(forms.FileField):
    widget = MultiFileInput

    def to_python(self, data):
        # todo: test me
        data = [d for d in data if d not in EMPTY_VALUES]
        try:
            return [self._get_file_path_from_sha256(sha256_of_file) for sha256_of_file in data]
        except Exception:
            raise ValidationError(_('files_uploading_error'))

    @retry(exceptions=(ProxyStorageModel.DoesNotExist, ), attempts=20, delay=0.1)
    def _get_file_path_from_sha256(self, sha256_of_file):
        file_meta_info = ProxyStorageModel.objects.get_by_sha256(sha256_of_file, self.question.survey_id)
        return file_meta_info.path

    def clean(self, data, initial=None):
        # todo: test me
        data = super(MultiFileField, self).clean(data, initial=initial)
        if isinstance(data, str):
            data = json.loads(data)
        return data
