# -*- coding: utf-8 -*-
import dateutil
import datetime

from django.conf import settings
from events.common_app.utils import get_user_ip_address, requests_session
from events.yauth_contrib.auth import TvmAuth


def get_alert_id(token_name):
    return f'{settings.SOLOMON_SERVICE}_{settings.SOLOMON_CLUSTER}_{token_name}'


class Tokens:
    never_expired = 1000

    def _get_expired_in_days(self, oauth_token):
        bb = settings.INTERNAL_SITE_BLACKBOX_INSTANCE
        result = bb.oauth(oauth_token, userip=get_user_ip_address(), by_token=True)
        if result.get('status') == 'INVALID':
            return 0

        oauth = result.get('oauth')
        if not oauth:
            return None

        expire_time = oauth.get('expire_time', '')
        if not expire_time:
            return self.never_expired

        try:
            delta = dateutil.parser.parse(expire_time) - datetime.datetime.now()
            return delta.days if delta.days > 0 else 0
        except dateutil.parser.ParserError:
            return None

    def items(self):
        for key in dir(settings):
            if key.endswith('TOKEN'):
                yield key, getattr(settings, key, None)

    def get_expired_in_days(self):
        for name, value in self.items():
            days = self._get_expired_in_days(value)
            if days is not None:
                yield name, days


class SolomonMetrics:
    def __init__(self):
        self.api_url = f'{settings.SOLOMON_URL}/api/v2'

    def _get_auth(self):
        return TvmAuth(settings.SOLOMON_TVM2_CLIENT)

    def _make_request(self, path, method=None, data=None, params=None):
        url = f'{self.api_url}{path}'
        method = method or 'get'
        return requests_session.request(method, url, params=params, json=data, auth=self._get_auth())

    def _get_metrics(self, data):
        return [
            {
                'labels': {'sensor': label},
                'value': value,
            }
            for label, value in data
        ]

    def write_metrics(self, sensor_data):
        params = {
            'project': settings.SOLOMON_PROJECT,
            'cluster': settings.SOLOMON_CLUSTER,
            'service': settings.SOLOMON_SERVICE,
        }
        data = {
            'commonLabels': {'host': 'forms-push'},
            'metrics': self._get_metrics(sensor_data),
        }
        return self._make_request('/push/', method='post', data=data, params=params)


class SolomonAlerts:
    max_page_size = 100

    def __init__(self):
        self.api_url = f'{settings.SOLOMON_URL}/api/v2/projects/{settings.SOLOMON_PROJECT}'

    def _get_auth(self):
        return TvmAuth(settings.SOLOMON_TVM2_CLIENT)

    def _make_request(self, path, method=None, data=None, params=None):
        url = f'{self.api_url}{path}'
        method = method or 'get'
        return requests_session.request(method, url, params=params, json=data, auth=self._get_auth())

    def _get_token_alert_data(self, alert_id, sensor_name):
        return {
            'projectId': settings.SOLOMON_PROJECT,
            'id': alert_id,
            'name': alert_id,
            'channels': [
                {
                    'id': 'telegram',
                    'config': {
                        'notifyAboutStatuses': ['OK', 'ALARM', 'WARN'],
                        'repeatDelaySecs': 60 * 60,  # 1 hour
                    },
                },
            ],
            'type': {
                'expression': {
                    'program': f'''
let data = {{
    project="{settings.SOLOMON_PROJECT}", cluster="{settings.SOLOMON_CLUSTER}",
    service="{settings.SOLOMON_SERVICE}", sensor="{sensor_name}"
}};

no_data_if(count(data) == 0);
let value = last(data);
let fixedValue = to_fixed(value, 0);

alarm_if(value <= 2);
warn_if(value <= 7);
                    ''',
                    'checkExpression': '',
                },
            },
            'annotations': {
                'value': f'{alert_id} expired in {{{{expression.fixedValue}}}} days'
            },
            'windowSecs': 60 * 60,  # 1 hour
            'delaySecs': 0,
        }

    def create_token_alert(self, alert_id, sensor_name):
        alert_data = self._get_token_alert_data(alert_id, sensor_name)
        return self._make_request('/alerts/', method='post', data=alert_data)

    def modify_token_alert(self, alert_id, sensor_name, version):
        alert_data = self._get_token_alert_data(alert_id, sensor_name)
        alert_data['version'] = version
        return self._make_request(f'/alerts/{alert_id}/', method='put', data=alert_data)

    def create_or_modify_token_alert(self, token_name):
        alert_id = get_alert_id(token_name)
        alert_data = self.get_alert(alert_id)
        if alert_data:
            version = alert_data['version']
            return self.modify_token_alert(alert_id, token_name, version)
        else:
            return self.create_token_alert(alert_id, token_name)

    def make_token_alerts(self):
        tokens = Tokens()
        for token_name, _ in tokens.items():
            self.create_or_modify_token_alert(token_name)

    def get_alert(self, alert_id):
        response = self._make_request(f'/alerts/{alert_id}/')
        if response.status_code == 200:
            return response.json()

    def get_alerts(self):
        params = {'pageSize': self.max_page_size}
        response = self._make_request('/alerts/', params=params)
        while response.status_code == 200:
            data = response.json()
            for it in data.get('items') or []:
                yield it

            next_token = data.get('nextPageToken')
            if not next_token:
                break

            params['pageToken'] = next_token
            response = self._make_request('/alerts/', params=params)
