# -*- coding: utf-8 -*-
import logging

from django.utils import translation
from django.conf import settings
from django.utils.deprecation import MiddlewareMixin
from django.utils.translation.trans_real import parse_accept_lang_header
from django.http import HttpResponse
from threading import local


logger = logging.getLogger(__name__)


class PingMiddleware(MiddlewareMixin):
    def process_request(self, request):
        if request.path == '/v1/ping/':
            return HttpResponse('')


class DisableCSRFProtection(MiddlewareMixin):
    def process_request(self, request):
        setattr(request, '_dont_enforce_csrf_checks', True)


class OrgIdMiddleware(MiddlewareMixin):
    def process_request(self, request):
        orgs_in_request = request.META.get('HTTP_X_ORGS')

        if orgs_in_request and orgs_in_request != 'null':
            orgs_in_request = orgs_in_request.split(',')
            has_orgs = True
        else:
            orgs_in_request = []
            has_orgs = False

        request.orgs = orgs_in_request
        request.has_orgs = has_orgs


class EnableHTTPSMiddleware(MiddlewareMixin):
    def process_request(self, request):
        is_secure = getattr(settings, 'IS_SECURE', None)
        if is_secure is not None:
            request._get_scheme = lambda: 'https' if is_secure else 'http'


class FrontendAuthMiddleware(MiddlewareMixin):
    def process_request(self, request):
        request.frontend = settings.FRONTENDS_BY_AUTH_KEY.get(request.META.get('HTTP_X_FRONTEND_AUTHORIZATION'))


class SetRemoterAddresses(MiddlewareMixin):
    # todo: test me
    """
    django_yauth для blackbox'а использует значение из REMOTE_ADDR. При этом использования nginx в качестве реверс
    прокси может привести к тому, что REMOTE_ADDR получит значение 127.0.0.1
    С недавних пор паспорт отказывает в авторизации запросам, ip которых == 127.0.0.1
    """
    def process_request(self, request):
        from ylog.context import put_to_context
        request.META['FRONTEND_REMOTE_ADDR'] = request.META.get('HTTP_X_REAL_IP') or request.META['REMOTE_ADDR']
        if request.frontend and request.META.get('HTTP_X_USER_IP'):
            request.META['REMOTE_ADDR'] = request.META['HTTP_X_USER_IP']
        else:
            request.META['REMOTE_ADDR'] = request.META['FRONTEND_REMOTE_ADDR']
        user_ip = request.META.get('HTTP_X_FORWARDED_FOR', '').split(',')[0]
        put_to_context('user_ip', user_ip)


_thread_locals = local()


def get_current_request():
    """ returns the request object for this thead """
    # todo: test me
    return getattr(_thread_locals, 'request', None)


def get_current_request_body():
    """ returns the request object for this thead """
    # todo: test me
    return getattr(_thread_locals, 'body', None)


class ThreadLocalMiddleware(MiddlewareMixin):
    """ Simple middleware that adds the request object in thread local storage.

    Usage:
    >> from events.common_app.middlewares import get_current_request

    # Get the current request object:
    >> request = get_current_request()

    """
    # todo: test me
    def process_request(self, request):
        # todo: test me
        _thread_locals.request = request
        if request.path.startswith('/admin/'):
            _thread_locals.body = bytes(request.body)
        else:
            _thread_locals.body = bytes()


class RequestGetParamsToHeadersMiddleware(MiddlewareMixin):
    """Переписывает заголовки запроса на переданные GET параметры
    Ассоциации переписывания указаны в словаре REQUEST_GET_PARAMS_TO_HEADERS
    """
    # todo: test me
    def process_request(self, request):
        for key, meta_key in settings.REQUEST_GET_PARAMS_TO_HEADERS.items():
            if key in request.GET:
                request.META[meta_key] = request.GET[key]


class CheckSameLanguages(MiddlewareMixin):
    """В танкере для всех диалектов испанского, кроме Испании, язык хранится в es-la
    Пользователи приходят со своими языками.
    Эта миддлвара матчит язык пользователя в нужный по настройке SAME_LANGUAGES
    """
    # todo: test me
    def process_request(self, request):
        accept_lang = parse_accept_lang_header(request.META.get('HTTP_ACCEPT_LANGUAGE', ''))
        for lang in accept_lang:
            lower_lang = lang[0].lower()
            user_language = None
            if lower_lang in settings.SAME_LANGUAGES:
                user_language = settings.SAME_LANGUAGES[lower_lang]
            elif lower_lang in settings.SAME_LANGUAGES.values():
                user_language = lower_lang
            if user_language:
                translation.activate(user_language)
                return
