# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django.db import models
from django.db.models.query import QuerySet
from django.db.utils import IntegrityError

from events.common_app.conditions import PublishedConditions
from events.common_app.utils import requests_session


logger = logging.getLogger(__name__)


class PublishedQuerySet(QuerySet):
    def filter_for_current_site_type(self):
        """Фильтрует сущности под текущий тип сайта (для yandex.ru или yandex-team.ru пользователей)"""
        return self.filter(PublishedConditions.is_for_current_site_type())


class PublishedManager(models.Manager):
    def get_queryset(self):
        return PublishedQuerySet(model=self.model, using=self._db)

    def filter_for_current_site_type(self):
        return self.get_queryset().filter_for_current_site_type()


class PublishedMixin(models.Model):
    # опубликовано для внешних пользователей
    is_published_external = models.BooleanField(default=False)
    # опубликовано для внутренних пользователей
    is_published_internal = models.BooleanField(default=True, editable=False)

    objects = PublishedManager()

    class Meta:
        abstract = True

    @property
    def is_for_current_site_type(self):
        return PublishedConditions.is_for_current_site_type(model_instance=self)


class TimeStampMixin(models.Model):
    date_created = models.DateTimeField(auto_now_add=True, db_index=True)  # дата создания
    date_updated = models.DateTimeField(auto_now=True)  # дата изменения

    class Meta:
        abstract = True


class SyncWithSourceManagerMixin(models.Manager):
    def sync_with_source(self, source_url):
        source_data = requests_session.request('GET', source_url).json()
        for source_element in source_data:
            model_data = self._get_model_data_from_source(source_element)
            try:
                model, _ = self.get_or_create(**model_data)
            except (IntegrityError, TypeError, ValueError):
                logger.exception('Got error while synchronizing with source')

    def _get_model_data_from_source(self, source_element):
        model_data = dict()
        for source_field_name, model_field_name in self.model.MetaSource.source_to_model_fields.items():
            if source_field_name in source_element:
                model_data[model_field_name] = source_element[source_field_name]
        settings_languages = [l[0] for l in settings.LANGUAGES]
        for language in source_element['title'].keys():
            if language in settings_languages:
                model_data['title_%s' % language] = source_element['title'][language]
            if language == settings.MODELTRANSLATION_DEFAULT_LANGUAGE:
                model_data['title'] = source_element['title'][language]

        return model_data
