# -*- coding: utf-8 -*-
import logging
import itertools

from django.conf import settings
from django.core.cache import caches
from django.utils.translation import ugettext as _
from requests.exceptions import RequestException

from events.common_app.utils import requests_session
from events.surveyme_integration.exceptions import TrackerConfigError
from events.yauth_contrib.auth import TvmAuth, OAuth

logger = logging.getLogger(__name__)


def cacheit(name: str, *parts):
    not_found = object()
    def outter(func):
        def inner(self, *args, **kwargs):
            cache = caches['startrek']
            sargs = map(str, args)
            cache_key = ':'.join((name, self.dir_id or '', self.lang, *sargs))[:100]
            result = cache.get(cache_key, not_found)
            if result is not_found:
                result = func(self, *args, **kwargs)
                cache.set(cache_key, result)
            return result
        return inner
    return outter


class StartrekClient:
    def __init__(self, uid, lang=None, dir_id=None):
        self.uid = uid
        self.lang = lang or 'ru'
        self.dir_id = dir_id
        self.auth = TvmAuth(settings.TRACKER_TVM2_CLIENT)

    def _make_request(self, method, path, params=None, **kwargs):
        params = params or {}
        params.update({
            'uid': self.uid,
            'language': self.lang,
        })
        headers = {
            'User-Agent': settings.STARTREK_USERAGENT,
        }
        if settings.IS_BUSINESS_SITE:
            headers['X-Org-Id'] = self.dir_id

        auth = kwargs.pop('auth', None) or self.auth
        url = f'{settings.STARTREK_API_HOST}{path}'
        logger.info('Tracker request url %s, params %s, body %s, headers %s', url, params, kwargs, headers)
        response = requests_session.request(
            method.lower(),
            url,
            params=params,
            headers=headers,
            auth=auth,
            timeout=settings.STARTREK_TIMEOUT,
            verify=settings.YANDEX_ROOT_CERTIFICATE,
            **kwargs,
        )
        response.raise_for_status()
        return response

    def _get(self, path, **kwargs):
        try:
            return self._make_request('get', path, **kwargs).json()
        except RequestException:
            pass

    def _post(self, path, **kwargs):
        return self._make_request('post', path, **kwargs).json()

    def _patch(self, path, **kwargs):
        return self._make_request('patch', path, **kwargs).json()

    def create_queue(self, name, key, description,
                     default_type='task', default_priority='normal',
                     lead=None, issue_types_config=None):
        issue_type_config = {
            'issueType': default_type,
            'workflow': 'oicn',
            'resolutions': ['won\'tFix'],
        }
        if settings.IS_BUSINESS_SITE:
            issue_type_config.update({
                'workflow': 'common',
                'resolutions': ['wontFix'],
            })
        issue_types_config = issue_types_config or [issue_type_config]
        lead = lead or self.uid
        fields = {
            'name': name,
            'key': key,
            'lead': lead,
            'description': description,
            'defaultType': default_type,
            'defaultPriority': default_priority,
            'issueTypesConfig': issue_types_config
        }
        return self._post('/service/queues/', json=fields)

    def create_attachment(self, file_name, file_content):
        params = {
            'filename': file_name,
        }
        files = [
            ('files', (file_name, file_content, None)),
        ]
        return self._post('/service/attachments/', params=params, files=files)

    def create_issue(self, data):
        parent = data.get('parent')
        if not data.get('queue') and parent:
            data['queue'] = parent.split('-', 1)[0]

        if not data.get('queue'):
            raise TrackerConfigError(_('Не указано имя очереди.'))

        params = {
            'notifyAuthor': True,
        }
        return self._post('/service/issues/', params=params, json=data)

    def update_issue(self, issue_key, data):
        return self._patch(f'/service/issues/{issue_key}/', json=data)

    def get_issue_by_unique_value(self, unique_value):
        params = {
            'filter': f'unique:{unique_value}',
            'perPage': 1,
        }
        result = self._get('/service/issues/', params=params)
        if result:
            return result[0]

    @cacheit('issues')
    def get_issue(self, issue_key):
        return self._get(f'/service/issues/{issue_key}/')

    @cacheit('queues')
    def get_queue(self, queue_name):
        return self._get(f'/service/queues/{queue_name}/')

    @cacheit('priorities')
    def get_priorities(self):
        return self._get('/service/priorities/')

    @cacheit('priorities')
    def get_priority(self, priority):
        return self._get(f'/service/priorities/{priority}/')

    @cacheit('fields')
    def get_fields(self):
        return self._get('/service/fields/')

    @cacheit('localFields')
    def get_local_fields(self, queue_name):
        auth = OAuth(settings.STARTREK_OAUTH_ACCESS_TOKEN)
        return self._get(f'/v2/queues/{queue_name}/localFields/', auth=auth)

    def _get_system_field(self, field_id):
        return self._get(f'/service/fields/{field_id}/')

    def _get_local_field(self, field_id):
        return self._get(f'/service/localFields/{field_id}/')

    @cacheit('field')
    def get_field(self, field_id):
        if '--' in field_id:
            return self._get_local_field(field_id)
        else:
            return self._get_system_field(field_id)

    @cacheit('types')
    def get_issuetypes(self, queue_name):
        return self._get(f'/service/queues/{queue_name}/issuetypes/')

    @cacheit('types')
    def get_issuetype(self, issuetype):
        return self._get(f'/service/issuetypes/{issuetype}/')

    @cacheit('components')
    def get_components(self, queue_name):
        return self._get(f'/service/queues/{queue_name}/components/')

    def get_attachments(self, issue_key):
        return self._get(f'/service/issues/{issue_key}/attachments/')


class StartrekFieldsSuggest:
    def __init__(self, dir_id=None, lang=None):
        self.client = get_startrek_client(dir_id=dir_id, lang=lang)

    def suggest(self, search, queue_name=None):
        return [
            field
            for field in self._get_fields(queue_name)
            if self._is_matched(field, search)
        ]

    def find(self, field_id, queue_name=None):
        for field in self._get_fields(queue_name):
            if field['id'] == field_id:
                return field

    def find_many(self, field_ids, queue_name=None):
        fields_by_id = {
            field['id']: field
            for field in self._get_fields(queue_name)
        }
        for field_id in field_ids:
            field = fields_by_id.get(field_id)
            if field:
                yield field

    def _get_fields(self, queue_name=None):
        fields = []
        if queue_name and not settings.IS_BUSINESS_SITE:
            fields.append(self.client.get_local_fields(queue_name))
        fields.append(self.client.get_fields())
        return itertools.chain(*fields)

    def _is_matched(self, field, search):
        slug = (field.get('key') or field.get('id')).lower()
        if slug.find(search) != -1:
            return True

        name = field['name'].lower()
        return name.find(search) != -1


def get_robot_tracker(org_dir_id):
    from events.common_app.directory import CachedDirectoryClient
    client = CachedDirectoryClient()
    return client.get_service_user(org_dir_id, 'tracker')


def get_kwargs_for_business(dir_id):
    return {
        'uid': get_robot_tracker(dir_id),
        'dir_id': dir_id,
    }


def get_kwargs_for_intranet():
    return {
        'uid': settings.FORMS_ROBOT_UID,
    }


def get_startrek_auth_kwargs(dir_id=None):
    if settings.IS_BUSINESS_SITE:
        return get_kwargs_for_business(dir_id)
    else:
        return get_kwargs_for_intranet()


def get_startrek_client(dir_id=None, lang=None):
    auth_kwargs = get_startrek_auth_kwargs(dir_id=dir_id)
    return StartrekClient(lang=lang, **auth_kwargs)
