import re
import requests

from urllib.parse import urlparse
from django.conf import settings

from events.common_app.tvm2_client import get_service_ticket
from events.yauth_contrib.auth import TvmAuth


class ZoraSession(requests.Session):
    def __init__(self, request_id=None):
        super(ZoraSession, self).__init__()
        self.request_id = request_id

    def get_zora_context(self, **kwargs):
        headers = kwargs.pop('headers', {})
        if self.request_id is not None:
            headers['X-Ya-Req-Id'] = self.request_id
        headers['X-Ya-User-Agent'] = 'Yandex.Forms/1.0'
        kwargs['headers'] = headers
        source = settings.GO_ZORA_SOURCE_NAME
        tvm_ticket = get_service_ticket(settings.GO_ZORA_TVM2_CLIENT)
        zora_url = f'http://{source}:{tvm_ticket}@{settings.GO_ZORA_URL}:{settings.GO_ZORA_PORT}/'
        kwargs['proxies'] = {
            'http': zora_url,
            'https': zora_url,
        }
        return kwargs

    def request(self, method, url, **kwargs):
        kwargs = self.get_zora_context(**kwargs)
        kwargs['auth'] = TvmAuth(settings.GO_ZORA_TVM2_CLIENT)
        kwargs['verify'] = False
        kwargs['timeout'] = settings.GO_ZORA_TIMEOUT
        return super(ZoraSession, self).request(method, url, **kwargs)


class Session(requests.Session):
    def request(self, method, url, **kwargs):
        kwargs['verify'] = settings.YANDEX_ROOT_CERTIFICATE
        timeout = kwargs.pop('timeout', None)
        kwargs['timeout'] = timeout or settings.DEFAULT_TIMEOUT
        return super(Session, self).request(method, url, **kwargs)


def is_external_host(url):
    domain_name = urlparse(url).netloc
    if settings.IS_BUSINESS_SITE:
        return domain_name not in settings.B2B_WHITE_LIST

    domain_re = re.compile(r'\.yandex(-team)?\.(net|ru)(:\d+)?')
    return not bool(domain_re.search(domain_name))


def get_session(url, request_id=None):
    if is_external_host(url):
        return ZoraSession(request_id)
    return Session()
