# -*- coding: utf-8 -*-
import os
import logging

from urllib.parse import quote_plus
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.http import Http404, HttpResponse
from django.core.exceptions import SuspiciousFileOperation

from rest_framework import views, status
from rest_framework.exceptions import PermissionDenied, NotAuthenticated
from rest_framework.response import Response
from ylog.context import log_context

from events.common_app.utils import slugify_filename
from events.common_storages.models import ProxyStorageModel
from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.storage import ReadError
from events.surveyme.models import Survey
from events.rest_framework_contrib.mixins import ExternalGenericApiViewV1Mixin
from events.rest_framework_contrib.permissions import ANY, HasFilePermission, TakeoutClientOnly

logger = logging.getLogger(__name__)


class FileView(ExternalGenericApiViewV1Mixin, views.APIView):
    create_on_missing = False
    permission_classes = [ANY(HasFilePermission, TakeoutClientOnly)]
    types = {
        'txt': 'text/plain;charset=utf-8',
        'gif': 'image/gif',
        'jpeg': 'image/jpeg',
        'jpg': 'image/jpeg',
        'png': 'image/png',
        'bmp': 'image/bmp',
        'tiff': 'image/tiff',
        'tif': 'image/tiff',
        'pdf': 'application/pdf',
        'jfif': 'image/jpeg',
        'pjpeg': 'image/jpeg',
        'pjp': 'image/jpeg',
        'apng': 'image/apng',
        'ico': 'image/x-icon',
        'cur': 'image/x-icon',
        'webp': 'image/webp',

    }
    default_type = 'application/octet-stream'

    def get(self, request, *args, **kwargs):
        file_path = request.query_params.get('path')
        with log_context(file_path=file_path):
            try:
                file_info = (
                    ProxyStorageModel.objects.using(settings.DATABASE_ROLOCAL)
                    .select_related('survey')
                    .get(path=file_path)
                )
            except ProxyStorageModel.DoesNotExist:
                raise Http404

            try:
                self.check_object_permissions(self.request, file_info)
            except (PermissionDenied, NotAuthenticated):
                logger.info("User does't have access to this file")
                raise Http404

            # ok, lets serve file
            file_name = self._get_file_name(file_info)
            content_type = self._get_content_type(file_name)
            response = HttpResponse(content_type=content_type)
            if request.query_params.get('download') == '1' or content_type == self.default_type:
                response['Content-Disposition'] = 'attachment; filename="{0}"'.format(quote_plus(file_name))

            response['X-Content-Type-Options'] = 'nosniff'

            storage = file_info.get_original_storage()
            try:
                response.write(storage.open(file_path).read())
            except SuspiciousFileOperation:
                logger.warning("Got SuspiciousFileOperation while working with file")
                raise Http404
            except ReadError:
                logger.warning('File %s not found', file_path)
                raise Http404
            return response

    def _get_file_name(self, file_info):
        file_name = file_info.original_name
        if not file_name:
            # на случай если не заполнено поле original_name, справедливо для тестов
            # преобразование /403/d96d215163dd69e8f6935a894c3fdc46_russian.txt => russian.txt
            file_name = file_info.path.split('_', 1)[-1]  # берем значение за первым `_`
            # преобразование /hello/world.txt => world.txt
            file_name = file_name.split('/')[-1]  # для тестов, берем значение за последним `/`
        return file_name

    def _get_content_type(self, file_name):
        ext = os.path.splitext(file_name)[1].strip('.')
        return self.types.get(ext, self.default_type)

    def get_survey_id(self, request):
        survey_id = request.query_params.get('survey')
        if not settings.IS_BUSINESS_SITE:
            if survey_id and not survey_id.isdigit():
                try:
                    survey = (
                        Survey.objects.using(settings.DATABASE_ROLOCAL)
                        .values_list('pk', named=True)
                        .get(slug=survey_id)
                    )
                    return survey.pk
                except Survey.DoesNotExist:
                    return None
        return survey_id

    def post(self, request, *args, **kwargs):
        # forms-17
        if not request.frontend:
            raise PermissionDenied
        if settings.IS_BUSINESS_SITE:
            expire = settings.BUSINESS_EXPIRE_TIMEOUT
        else:
            expire = settings.DEFAULT_EXPIRE_TIMEOUT
        survey_id = self.get_survey_id(request)
        user_id = None
        if not request.user.is_anonymous:
            user_id = request.user.pk
        proxy_storage = ProxyStorage(expire=expire)
        file_obj = request.data.get('file')
        if not file_obj or not file_obj.size or not file_obj.name:
            return Response({'errors': [_('Файл не может быть пустым')]}, status=status.HTTP_400_BAD_REQUEST)
        if file_obj.size > settings.MDS_MAX_FILE_SIZE:
            return Response({'errors': [_('Слишком большой файл')]}, status=status.HTTP_400_BAD_REQUEST)
        name = slugify_filename(file_obj.name)
        path = proxy_storage.save(name, file_obj, user_id=user_id, survey_id=survey_id)
        file_meta_info = ProxyStorageModel.objects.get(path=path)
        file_info = {
            'id': file_meta_info.sha256,
            'name': name,
        }
        return Response(data=file_info)
