# -*- coding: utf-8 -*-
from django.conf import settings
from django.utils.translation import ugettext as _
from rest_framework import views, status
from rest_framework.response import Response

from events.common_app.utils import get_upload_to_translified_func_for_path
from events.common_storages.proxy_storages import ProxyStorage
from events.common_storages.models import ProxyStorageModel
from events.rest_framework_contrib.mixins import InternalGenericApiViewV2Mixin


class FileView(InternalGenericApiViewV2Mixin, views.APIView):
    def post(self, request, *args, **kwargs):
        file_obj = request.data.get('file')

        if not file_obj or not file_obj.size or not file_obj.name:
            return Response({'errors': [_("The submitted file is empty.")]}, status=status.HTTP_400_BAD_REQUEST)

        survey_id = request.query_params.get('survey')
        user_id = None
        if not request.user.is_anonymous:
            user_id = request.user.pk

        proxy_storage = ProxyStorage()
        name = self._get_file_name(request, file_obj)
        path = proxy_storage.save(name, file_obj, user_id=user_id, survey_id=survey_id)
        meta_info = ProxyStorageModel.objects.get(path=path)
        file_info = {
            'content_uri': proxy_storage.url(path, site_type='internal'),
            'path': path,
            'file_size': meta_info.file_size,
            'original_name': meta_info.original_name,
            'namespace': meta_info.namespace or settings.MDS_OLD_NAMESPACE,
        }
        return Response(data=file_info)

    def _get_file_name(self, request, file_obj):
        upload_to = request.data.get('upload_to', '')
        if upload_to:
            upload_to = '%s_%s' % ('admin', upload_to)
        else:
            upload_to = 'admin'
        return get_upload_to_translified_func_for_path(upload_to)(file_obj, file_obj.name)
