# -*- coding: utf-8 -*-
import datetime
import json
import logging
import operator

from dateutil import parser
from django.db import models
from django.core.exceptions import ValidationError
from django.contrib.contenttypes.models import ContentType

from events.common_app.fields import JSONField
from events.common_app.utils import flatlist

logger = logging.getLogger(__name__)

OPERATOR_CHOICES = (
    ('and', 'и'),
    ('or', 'или'),
)

BASE_CONDITION_CHOICES = (
    ('eq', 'равно'),
    ('neq', 'не равно'),
)

CONDITION_CHOICES = BASE_CONDITION_CHOICES + (
    ('lt', 'меньше'),
    ('gt', 'больше'),
)


class ContentTypeAttribute(models.Model):
    title = models.CharField(max_length=100)
    content_type = models.ForeignKey(ContentType, on_delete=models.DO_NOTHING)
    attr = models.CharField(max_length=255)
    lookup_field = models.CharField(max_length=255, default='pk')
    allowed_conditions = JSONField(default=json.dumps([e[0] for e in BASE_CONDITION_CHOICES]))

    class Meta:
        unique_together = (('content_type', 'attr', 'lookup_field'),)

    def apply_condition(self, obj, condition, value):
        list_value = self._get_value_to_compare_with_exp_value(obj)
        return self.apply(condition, list_value, value)

    def _is_empty_list_value(self, list_value):
        if not list_value or (len(list_value) == 1 and (list_value[0] is None or list_value[0] == '')):
            return True
        return False

    def apply(self, condition, list_value, exp_value):
        if not isinstance(list_value, list):
            list_value = [list_value]
        if self._is_empty_list_value(list_value):
            return self._apply_empty_value(condition, exp_value)
        if self.attr == 'answer_choices':
            result = self._apply_choices_condition(condition, list_value, exp_value)
        elif self.attr == 'answer_boolean':
            result = self._apply_boolean_condition(condition, list_value, exp_value)
        elif self.attr.startswith('answer_date'):
            result = self._apply_date_condition(condition, list_value, exp_value)
        elif self.attr == 'answer_number':
            result = self._apply_number_condition(condition, list_value, exp_value)
        else:
            result = self._apply_text_condition(condition, list_value, exp_value)
        return result

    def _apply_empty_value(self, condition, exp_value):
        result = exp_value == '' or exp_value is None
        if condition == 'eq':
            return result
        elif condition == 'neq':
            return not result
        return False

    def _get_value_to_compare_with_exp_value(self, obj):
        from events.surveyme.models import ProfileSurveyAnswer
        if isinstance(obj, ProfileSurveyAnswer):
            return getattr(obj, self.attr, None)
        return obj

    def _apply_choices_condition(self, condition, list_value, exp_value):
        values = set()
        for value in flatlist(list_value):
            if isinstance(value, dict):
                value = value.get('key')
            values.add(value)
        if condition == 'eq':
            return exp_value in values
        elif condition == 'neq':
            return exp_value not in values
        return False

    def _apply_boolean_condition(self, condition, list_value, exp_value):
        exp_value = str(exp_value)
        for value in list_value:
            value = str(value)
            if condition == 'eq':
                if value == exp_value:
                    return True
            elif condition == 'neq':
                if value != exp_value:
                    return True
        return False

    def _apply_date_condition(self, condition, list_value, exp_value):
        exp_value = self._get_date_object_or_none(exp_value)
        for value in list_value:
            value = self._get_date_object_or_none(value)
            if not value:
                continue
            if condition == 'eq':
                if value == exp_value:
                    return True
            elif condition == 'neq':
                if value != exp_value:
                    return True
            elif condition == 'gt':
                if exp_value and value > exp_value:
                    return True
            elif condition == 'lt':
                if exp_value and value < exp_value:
                    return True
        return False

    def _get_date_object_or_none(self, value):
        if not value:
            return None
        if isinstance(value, datetime.datetime):
            return value.date()
        if isinstance(value, datetime.date):
            return value
        if isinstance(value, dict):
            if self.attr.endswith('start'):
                value = value.get('begin')
            elif self.attr.endswith('end'):
                value = value.get('end')
        try:
            return parser.parse(value).date()
        except (ValueError, OverflowError):
            return None

    def _apply_text_condition(self, condition, list_value, exp_value):
        for value in list_value:
            if condition == 'eq':
                if value == exp_value:
                    return True
            elif condition == 'neq':
                if value != exp_value:
                    return True
        return False

    def _get_number_object_or_none(self, value):
        try:
            return int(value)
        except (ValueError, TypeError):
            return None

    def _apply_number_condition(self, condition, list_value, exp_value):
        exp_value = self._get_number_object_or_none(exp_value)
        for value in list_value:
            value = self._get_number_object_or_none(value)
            if condition == 'eq':
                if value == exp_value:
                    return True
            elif condition == 'neq':
                if value != exp_value:
                    return True
            elif condition == 'gt':
                if value > exp_value:
                    return True
            elif condition == 'lt':
                if value < exp_value:
                    return True
        return False


class ConditionNodeBase(models.Model):
    position = models.PositiveSmallIntegerField(default=1)

    class Meta:
        abstract = True
        ordering = ('position',)

    def is_true(self, **kwargs):
        state = None
        for item in self.items.all():
            item_state = item.is_true(**kwargs)
            if state is None:
                state = bool(item_state)
            else:
                method = getattr(operator, '%s_' % item.operator.lower())
                state = method(state, bool(item_state))
        return bool(state)


class ConditionItemBase(models.Model):
    # node = models.ForeignKey(ConditionNodeBase, related_name='items', on_delete=models.DO_NOTHING)  # you must add node FK
    operator = models.CharField(choices=OPERATOR_CHOICES, default=OPERATOR_CHOICES[0][0], max_length=100)
    content_type_attribute = models.ForeignKey(ContentTypeAttribute, on_delete=models.CASCADE)
    condition = models.CharField(choices=CONDITION_CHOICES, default=CONDITION_CHOICES[0][0], max_length=100)
    value = models.CharField(max_length=100, blank=True)
    position = models.PositiveSmallIntegerField(default=1)

    class Meta:
        abstract = True
        ordering = ('position',)

    def is_true(self, **kwargs):
        return self.content_type_attribute.apply_condition(
            obj=self.get_obj(**kwargs),
            condition=self.condition,
            value=self.value
        )

    def get_obj(self, **kwargs):
        raise NotImplementedError

    def clean(self):
        self.validate_condition_is_allowed()

    def validate_condition_is_allowed(self):
        if self.condition not in self.content_type_attribute.allowed_conditions:
            raise ValidationError('Условие не может быть сохранено.')
