# -*- coding: utf-8 -*-
from django.forms.fields import Field
from django.forms.widgets import Widget
from django.core import validators
from django.core.exceptions import ValidationError
from django.utils.datastructures import MultiValueDict
from django.utils.translation import ugettext as _

from events.data_sources.models import DataSourceItem
from events.data_sources.sources import data_sources_by_name


class DataSourceFieldWidget(Widget):
    # todo: test me
    def value_from_datadict(self, data, files, name):
        if isinstance(data, MultiValueDict):
            return data.getlist(name)
        return data.get(name, None)

    def render(self, name, value, attrs=None):
        # todo: test me
        if value:
            return '<br>'.join([i['text'] for i in value])
        else:
            return ''


class DataSourceField(Field):
    widget = DataSourceFieldWidget

    def __init__(self, *args, **kwargs):
        # todo: test me
        self.data_source = kwargs.pop('data_source', None)
        self.allow_multiple_choice = kwargs.pop('allow_multiple_choice', False)
        self.filters = kwargs.pop('filters', {})
        self.user = kwargs.pop('user', None)
        self.language = kwargs.pop('language', None)
        self.default_error_messages.update({'invalid': _('Enter a valid value.')})
        super(DataSourceField, self).__init__(*args, **kwargs)

    def prepare_value(self, value):
        form = getattr(self, 'form')
        if value and form:
            if form.is_bound:
                # value is DataSourceItem identities
                values_qs = DataSourceItem.objects.filter(
                    identity__in=value,
                    data_source=self.data_source
                )
            else:
                # value is DataSourceItem ids
                values_qs = DataSourceItem.objects.filter(id__in=value)
            value = [
                {
                    'id': i.identity,
                    'text': i.get_text(),
                }
                for i in values_qs
            ]
        return value

    def validate(self, value):
        super(DataSourceField, self).validate(value)
        keys = list(value.keys())
        if keys:
            filter_data = {
                'id': keys,
                'user_uid': getattr(self.form.survey.user, 'uid', None),
                'dir_id': getattr(self.form.survey.org, 'dir_id', None),
            }
            filter_data.update(self.filters)
            items_found_in_queryset = []
            for item in self.get_filtered_data_source_data(filter_data=filter_data):
                if item['id'] in value:
                    value[item['id']] = item['text']
                    items_found_in_queryset.append(item['id'])
            inconsistent = set(keys) - set(items_found_in_queryset)
            if inconsistent:
                raise ValidationError(self.default_error_messages['invalid'])

    def get_filtered_data_source_data(self, filter_data):
        # todo: test me
        data_source_instance = self.get_data_source_instance()
        return data_source_instance.serializer_class(
            data_source_instance.get_filtered_queryset(filter_data=filter_data), many=True
        ).data

    def get_data_source_instance(self):
        # todo: test me
        return data_sources_by_name[self.data_source](
            user=self.user,
            language=self.language,
        )

    def to_python(self, value):
        value = value or []
        value = [v for v in value if v not in validators.EMPTY_VALUES]
        if not self.allow_multiple_choice:
            value = value[:1]
        return {val: None for val in value}

    def clean(self, value):
        value = super(DataSourceField, self).clean(value)
        return self.get_or_create_data_source_items(value)

    def get_or_create_data_source_items(self, items):
        return DataSourceItem.objects.get_and_update_bulk(
            data_source=self.data_source,
            items=items,
        )


class DataSourceFormFilterDataMixin(object):
    # todo: test me
    def __init__(self, *args, **kwargs):
        self.filter_data = kwargs.pop('filter_data') or {}
        super(DataSourceFormFilterDataMixin, self).__init__(*args, **kwargs)
        for key, field in self.fields.items():
            if (isinstance(field, DataSourceField) and key in self.filter_data):
                field.filters = self.filter_data[key]


class AnswerMatrixDataSourceField(DataSourceField):
    def validate(self, value):
        super(DataSourceField, self).validate(value)
        rows, columns = self._get_rows_and_columns(value)
        if rows and columns:
            rows_texts = self._validate_matrix_title_type(rows, type='row')
            columns_texts = self._validate_matrix_title_type(columns, type='column')
            for v in value:
                row, column = v.split('_')
                value[v] = '"%s": %s' % (rows_texts[row], columns_texts[column])

    def _get_rows_and_columns(self, value):
        rows = []
        columns = []
        for v in value:
            try:
                row, column = v.split('_')
            except (ValueError, IndexError):
                raise ValidationError(self.default_error_messages['invalid'])
            rows.append(row)
            columns.append(column)
        return rows, columns

    def _validate_matrix_title_type(self, keys, type):
        filter_data = {
            'id': keys,
            'type': type,
        }
        filter_data.update(self.filters)
        keys_found_in_queryset = []
        texts = {}
        for item in self.get_filtered_data_source_data(filter_data=filter_data):
            if item['id'] in keys:
                texts[item['id']] = item['text']
                keys_found_in_queryset.append(item['id'])
        inconsistent = set(keys) - set(keys_found_in_queryset)
        if inconsistent:
            raise ValidationError(self.default_error_messages['invalid'])
        return texts

    def to_python(self, value):
        value = value or []
        value = [v for v in value if v not in validators.EMPTY_VALUES]
        return {val: None for val in value}
