# -*- coding: utf-8 -*-
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ValidationError
from django.core.paginator import Page
from django.db.models import QuerySet
from django.utils.encoding import force_str
from rest_framework import serializers

from events.common_app.middleware import get_current_request
from events.common_app.utils import get_attr_or_key, get_lang_with_fallback


class BaseDataSourceCategory(object):
    name = None
    title = None


class BaseDataSource(object):
    name = None
    title = None
    desc = None
    category = None
    filter_class = None
    is_with_pagination = False
    is_need_auth = False
    required_filters = []
    allow_external_usage = False
    serializer_class = None
    is_proxy_data_source = False
    content_type_model = None
    is_for_external_site_type = True
    hint = True
    connect_only = False
    allow_empty_filter = False
    max_filtered_result = 50

    @classmethod
    def get_view_class(cls):
        from events.data_sources.api.views import DataSourceBaseViewSet
        return DataSourceBaseViewSet

    def __init__(self, user=None, language=None, request=None):
        self.user = user
        self.language = language
        self.request = request

    def get_filtered_queryset(self, filter_data=None):
        try:
            qs = self.filter_queryset(filter_data, queryset=self.get_queryset())
            if self.max_filtered_result:
                return qs[:self.max_filtered_result]
            return qs
        except ValueError as e:
            raise ValidationError(str(e))

    def filter_queryset(self, filter_data, queryset):
        if self.filter_class and filter_data:
            return self.filter_class(filter_data, queryset=queryset).qs
        elif self.allow_empty_filter:
            return queryset
        else:
            if isinstance(queryset, QuerySet):
                return queryset.none()
            else:
                return []

    def get_queryset(self):
        raise NotImplementedError()

    def get_content_type_id(self):
        # todo: test me
        content_type = self.get_content_type()
        if content_type:
            return content_type.id

    def get_content_type(self):
        # todo: test me
        if self.content_type_model:
            return ContentType.objects.get_for_model(self.content_type_model)

    def get_host(self):
        request = get_current_request()
        if request:
            return request.get_host()
        return 'localhost'

    def get_uri(self):
        # todo: test me
        return 'https://{host}/v1/data-source/{name}/'.format(
            host=self.get_host(),
            name=self.name.replace('_', '-'),
        )

    def get_admin_uri(self):
        # todo: test me
        if self.is_with_pagination:
            return 'https://{host}/admin/api/v2/data-source/{name}/'.format(
                host=self.get_host(),
                name=self.name.replace('_', '-'),
            )

    def get_admin_items(self):
        # todo: test me
        if not self.is_with_pagination and not self.is_need_auth and not self.required_filters:
            return self.serializer_class(self.get_filtered_queryset(), many=True).data

    def get_filters(self):
        # todo: test me
        if self.filter_class:
            response = []
            for key, value in self.filter_class.declared_filters.items():
                if hasattr(value, 'data_source'):
                    response.append({
                        'data_source': getattr(value, 'data_source'),
                        'filter': key
                    })
            return response or None


class TranslationDataSourceSerializerMixin:
    def to_representation(self, obj):
        lang_from_request, fallback_lang = get_lang_with_fallback()
        if obj.translations and lang_from_request != 'ru':
            translated_data = {}
            for field, translation_data in obj.translations.items():
                field_translation = translation_data.get(lang_from_request)
                if not field_translation and fallback_lang:
                    field_translation = translation_data.get(fallback_lang)
                if field_translation:
                    translated_data[field] = field_translation
            if translated_data:
                obj.__dict__.update(translated_data)

        return super(TranslationDataSourceSerializerMixin, self).to_representation(obj)


class SimpleDataSourceSerializer(serializers.Serializer):
    # todo: test me
    id_attr = None
    text_attr = None

    def __init__(self, instance=None, many=False, **kwargs):
        self.object = instance
        self._data = kwargs.get('data')
        self.label = 'label'
        self.read_only = kwargs.get('read_only')
        self.write_only = False
        self.many = many
        self.source = None

    def to_representation(self, obj):
        if isinstance(obj, QuerySet):
            return self.get_list(obj)
        return self.get_item(obj)

    def get_item(self, obj):
        return {
            'id': force_str(get_attr_or_key(obj, self.id_attr)),
            'text': force_str(get_attr_or_key(obj, self.text_attr)),
        }

    @property
    def data(self):
        """
        Returns the serialized data on the serializer.
        """
        if self._data is None:
            obj = self.object

            if self.many is not None:
                many = self.many
            else:
                many = hasattr(obj, '__iter__') and not isinstance(obj, (Page, dict))
            if many:
                self._data = [self.to_representation(item) for item in obj.all()]
            else:
                self._data = self.to_representation(obj)
        return self._data

    def initialize(self, *args, **kwargs):
        # for pagination
        pass

    def get_list(self, obj):
        # for pagination
        return [self.to_representation(item) for item in obj.all()]
