# -*- coding: utf-8 -*-
import logging

from django.db.models.functions import Length

from events.data_sources.filters import ListOrCommaSeparatedStringListFilter
from events.data_sources.sources.base import SimpleDataSourceSerializer
from events.data_sources.sources.external_table.base import (
    BaseExternalTableDataSource,
    BaseExternalTableDataSourceFilter,
)
from events.data_sources.utils import get_table_name, get_table_id


logger = logging.getLogger(__name__)


class YTDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'name'
    display_attr = 'display_name'

    class Meta:
        fields = ('id', 'name', 'display_name')

    def get_item(self, obj):
        return {
            'id': str(obj.get(self.id_attr)),
            'text': str(obj.get(self.display_attr) or obj.get(self.text_attr)),
        }


class YTDataSourceFilter(BaseExternalTableDataSourceFilter):
    id = ListOrCommaSeparatedStringListFilter(name=YTDataSourceSerializer.id_attr)

    def get_table_data(self, path, cluster):
        from events.data_sources.models import TableRow
        key = get_table_id(cluster, path)
        return TableRow.objects.filter(table_identifier=key)

    def _get_data(self, url):
        from events.data_sources.models import TableRow
        cluster, path = get_table_name(url)
        if cluster and path:
            data = self.get_table_data(path, cluster)
            filter_by_question = self.filter_data.get(YTDataSource.name)
            if self.suggest:
                data = data.filter(text__icontains=self.suggest)
            if self.ids_passed:
                if isinstance(self.ids_passed, (list, tuple, )):
                    data = data.filter(pk__in=self.ids_passed)
                else:
                    data = data.filter(pk=self.ids_passed)

            if filter_by_question:
                try:
                    filter_value = TableRow.objects.get(pk=filter_by_question)
                except TableRow.DoesNotExist:
                    logger.warn('YT record not found (cluster=%s, path=%s, filter_data=%s)', cluster, path, self.filter_data)
                    return []
                data = data.filter(filter_by=filter_value.source_id)
            if self.suggest:
                data = data.order_by(Length('text'), 'text')
            else:
                data = data.order_by('text')
            return [
                {'id': row.id, 'name': row.text, 'display_name': row.display_text}
                for row in data[:50]
            ]

    @staticmethod
    def declared_filters():
        return {
            YTDataSource.name: YTDataSource
        }


class YTDataSource(BaseExternalTableDataSource):
    name = 'yt_table_source'
    data_source = name
    title = 'Таблицы YT'
    desc = 'Данные из произвольной таблицы YT'
    filter_class = YTDataSourceFilter
    serializer_class = YTDataSourceSerializer
