# -*- coding: utf-8 -*-
import django_filters
from events.data_sources.filters import (
    CharListFilter,
    ListFilterSet,
    ListOrCommaSeparatedStringFilter,
)
from events.data_sources.sources.base import (
    BaseDataSourceCategory,
    BaseDataSource,
    SimpleDataSourceSerializer,
    TranslationDataSourceSerializerMixin,
)
from events.geobase_contrib.models import Country, City
from events.rest_framework_contrib import queryset_filters
from events.common_app.maps import get_maps_suggest
from events.translation.filters import TranslationCharFilter


class GenericDataSourceCategory(BaseDataSourceCategory):
    name = 'generic'
    title = 'Общее'


class CountryDataSourceSerializer(TranslationDataSourceSerializerMixin, SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'full_name'

    class Meta:
        fields = ('id', 'full_name')


class CountryDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=CountryDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = TranslationCharFilter(field_name=CountryDataSourceSerializer.text_attr)


class CountryDataSource(BaseDataSource):
    name = 'country'
    title = 'Страны'
    desc = 'Список стран мира'
    category = GenericDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    filter_class = CountryDataSourceFilter
    serializer_class = CountryDataSourceSerializer

    def get_queryset(self):
        return Country.objects.order_by('full_name')


class CityDataSourceSerializer(TranslationDataSourceSerializerMixin, SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'full_name'

    class Meta:
        fields = ('id', 'full_name')


class CityDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=CityDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = TranslationCharFilter(field_name=CityDataSourceSerializer.text_attr)
    country = ListOrCommaSeparatedStringFilter()
    country.data_source = CountryDataSource.name


class CityDataSource(BaseDataSource):
    name = 'city'
    title = 'Города'
    desc = 'Список городов мира'
    category = GenericDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    filter_class = CityDataSourceFilter
    serializer_class = CityDataSourceSerializer

    def get_queryset(self):
        return City.objects.order_by('-profiles_count', 'full_name')


class UniversityDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'full_name'

    class Meta:
        fields = ('id', 'full_name')


class UniversityDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=UniversityDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = django_filters.CharFilter(field_name=UniversityDataSourceSerializer.text_attr)
    is_deleted = False  # django_filters.BooleanFilter()


class UniversityDataSource(BaseDataSource):
    name = 'university'
    title = 'Университеты'
    desc = 'Список университетов'
    category = GenericDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    is_need_auth = False
    is_for_external_site_type = False  # TODO уточнить что это за настройка
    serializer_class = UniversityDataSourceSerializer
    filter_class = UniversityDataSourceFilter
    hint = False

    def get_queryset(self):
        from events.data_sources.models import University
        return (
            University.objects.all()
            .annotate_full_name()
            .order_by('full_name')
        )


class AddressDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'name'

    class Meta:
        fields = ('id', 'name')


class AddressDataSourceFilter(ListFilterSet):
    id = CharListFilter(field_name=AddressDataSourceSerializer.id_attr)
    text = CharListFilter(field_name=AddressDataSourceSerializer.text_attr)


class AddressDataSource(BaseDataSource):
    name = 'address'
    title = u'Адрес'
    desc = u'Адрес'
    category = GenericDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    serializer_class = AddressDataSourceSerializer
    filter_class = AddressDataSourceFilter

    def get_queryset(self):
        if self.request:
            text = self.request.query_params.get('suggest')
            if text:
                suggest_data = get_maps_suggest(text=text)
                return [
                    {
                        'id': suggest_text,
                        'name': suggest_text,
                    }
                    for suggest_text in suggest_data
                ]
        return []
