# -*- coding: utf-8 -*-
from django.utils.encoding import force_str

import django_filters

from events.data_sources.filters import ListOrCommaSeparatedStringFilter
from events.common_app.utils import get_attr_or_key
from events.staff.models import (
    StaffPerson,
    StaffGroup,
    StaffOrganization,
    StaffOffice,
)
from events.data_sources.sources.base import (
    BaseDataSourceCategory,
    BaseDataSource,
    SimpleDataSourceSerializer,
    TranslationDataSourceSerializerMixin,
)
from events.rest_framework_contrib import queryset_filters
from events.translation.filters import TranslationCharFilter


class StaffDataSourceCategory(BaseDataSourceCategory):
    name = 'staff'
    title = 'Стафф'


class StaffGroupDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'staff_id'
    text_attr = 'name'

    class Meta:
        fields = ('staff_id', 'name')


class StaffGroupDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=StaffGroupDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = django_filters.CharFilter(field_name=StaffGroupDataSourceSerializer.text_attr)
    is_deleted = django_filters.BooleanFilter()


class StaffGroupDataSource(BaseDataSource):
    name = 'staff_group'
    title = 'Группа на стаффе'
    desc = 'Группа на стаффе'
    category = StaffDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    is_need_auth = True
    is_for_external_site_type = False
    serializer_class = StaffGroupDataSourceSerializer
    filter_class = StaffGroupDataSourceFilter

    def get_queryset(self):
        return StaffGroup.objects.filter(type='department', is_deleted=False)


class StaffLoginDataSourceSerializer(TranslationDataSourceSerializerMixin, SimpleDataSourceSerializer):
    id_attr = 'login'
    text_attr = 'login'

    class Meta:
        fields = ('login', 'login')

    def to_representation(self, obj):
        return {
            'id': force_str(get_attr_or_key(obj, self.id_attr)),
            'text': '%s %s (%s)' % (obj.get_last_name(), obj.get_first_name(), obj.login),
        }


class StaffLoginDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=StaffLoginDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = django_filters.CharFilter(field_name=StaffLoginDataSourceSerializer.text_attr)
    is_deleted = django_filters.BooleanFilter()
    is_dismissed = django_filters.BooleanFilter()
    group_id = ListOrCommaSeparatedStringFilter()
    group_id.data_source = StaffGroupDataSource.name


class StaffLoginDataSource(BaseDataSource):
    name = 'staff_login'
    title = 'Логин на стаффе'
    desc = 'Внутренний логин'
    category = StaffDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    is_need_auth = True
    is_for_external_site_type = False
    serializer_class = StaffLoginDataSourceSerializer
    filter_class = StaffLoginDataSourceFilter

    def get_queryset(self):
        return StaffPerson.objects.filter(is_dismissed=False, is_deleted=False)


class StaffOrganizationDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'staff_id'
    text_attr = 'name'

    class Meta:
        fields = ('staff_id', 'name')


class StaffOrganizationDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=StaffOrganizationDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = django_filters.CharFilter(field_name=StaffOrganizationDataSourceSerializer.text_attr)
    is_deleted = django_filters.BooleanFilter()


class StaffOrganizationDataSource(BaseDataSource):
    name = 'staff_organization'
    title = 'Организация на стаффе'
    desc = 'Организация на стаффе'
    category = StaffDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    is_need_auth = True
    is_for_external_site_type = False
    serializer_class = StaffOrganizationDataSourceSerializer
    filter_class = StaffOrganizationDataSourceFilter

    def get_queryset(self):
        return StaffOrganization.objects.filter(is_deleted=False)


class StaffOfficeDataSourceSerializer(TranslationDataSourceSerializerMixin, SimpleDataSourceSerializer):
    id_attr = 'staff_id'
    text_attr = 'name'

    class Meta:
        fields = ('staff_id', 'name')


class StaffOfficeDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=StaffOfficeDataSourceSerializer.id_attr)
    suggest = queryset_filters.OneValueMethodFilter(field_name='filter_by_fit_for_search_text')
    text = TranslationCharFilter(field_name=StaffOfficeDataSourceSerializer.text_attr)
    is_deleted = django_filters.BooleanFilter()


class StaffOfficeDataSource(BaseDataSource):
    name = 'staff_office'
    title = 'Офис на стаффе'
    desc = 'Офис на стаффе'
    category = StaffDataSourceCategory()
    allow_external_usage = True
    is_with_pagination = True
    is_need_auth = True
    is_for_external_site_type = False
    serializer_class = StaffOfficeDataSourceSerializer
    filter_class = StaffOfficeDataSourceFilter

    def get_queryset(self):
        return StaffOffice.objects.filter(is_deleted=False)
