# -*- coding: utf-8 -*-
from django.utils.encoding import force_str

import django_filters

from events.data_sources.filters import ListOrCommaSeparatedStringFilter
from events.common_app.utils import get_attr_or_key
from events.data_sources.sources.base import BaseDataSourceCategory, BaseDataSource, SimpleDataSourceSerializer
from events.media.api_admin.v2.serializers import ImageSerializer


class SurveyQuestionChoiceDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'label'
    slug_attr = 'slug'

    class Meta:
        fields = ('id', 'label', 'slug')

    def to_representation(self, obj):
        result = {
            'id': force_str(get_attr_or_key(obj, self.id_attr)),
            'text': obj.get_label(),
            'slug': force_str(get_attr_or_key(obj, self.slug_attr)),
            'label_image': None,
        }
        label_image = getattr(obj, 'label_image', None)
        if label_image:
            result['label_image'] = ImageSerializer(label_image).data
        return result


class SurveysDataSourceCategory(BaseDataSourceCategory):
    name = 'surveys'
    title = 'Опросы'


class SurveyQuestionDataSource(BaseDataSource):
    pass


class SurveyQuestionChoiceDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=SurveyQuestionChoiceDataSourceSerializer.id_attr)
    slug = ListOrCommaSeparatedStringFilter(field_name=SurveyQuestionChoiceDataSourceSerializer.slug_attr)
    text = django_filters.CharFilter(field_name=SurveyQuestionChoiceDataSourceSerializer.text_attr)
    question = django_filters.NumberFilter('survey_question')


class SurveyQuestionChoiceDataSource(BaseDataSource):
    name = 'survey_question_choice'
    title = 'Варианты ответа на вопрос'
    desc = 'Список вариантов ответа на вопрос'
    category = SurveysDataSourceCategory()
    allow_external_usage = True
    serializer_class = SurveyQuestionChoiceDataSourceSerializer
    filter_class = SurveyQuestionChoiceDataSourceFilter
    required_filters = ['question']
    allow_empty_filter = True
    max_filtered_result = None

    def get_queryset(self):
        from events.surveyme.models import SurveyQuestionChoice
        return (
            SurveyQuestionChoice.objects.all().
            select_related(
                'label_image',
            )
        )

    def get_filtered_queryset(self, *args, **kwargs):
        queryset = super(SurveyQuestionChoiceDataSource, self).get_filtered_queryset(*args, **kwargs)
        return queryset.filter(is_hidden=False)


class SurveyQuestionMatrixChoiceDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'id'
    text_attr = 'label'
    type_attr = 'type'
    position_attr = 'position'

    class Meta:
        fields = ('id', 'label', 'type', 'position')

    def to_representation(self, obj):
        return {
            'id': force_str(get_attr_or_key(obj, self.id_attr)),
            'text': obj.get_label(),
            'type': force_str(get_attr_or_key(obj, self.type_attr)),
            'position': get_attr_or_key(obj, self.position_attr),
        }


class SurveyQuestionMatrixChoiceDataSourceFilter(django_filters.FilterSet):
    id = ListOrCommaSeparatedStringFilter(field_name=SurveyQuestionMatrixChoiceDataSourceSerializer.id_attr)
    text = django_filters.CharFilter(field_name=SurveyQuestionMatrixChoiceDataSourceSerializer.text_attr)
    question = django_filters.NumberFilter('survey_question')
    type = django_filters.CharFilter(field_name=SurveyQuestionMatrixChoiceDataSourceSerializer.type_attr)


class SurveyQuestionMatrixChoiceDataSource(BaseDataSource):
    name = 'survey_question_matrix_choice'
    title = 'Матрица вариантов ответа на вопрос'
    desc = 'Матрица вариантов ответа на вопрос'
    category = SurveysDataSourceCategory()
    allow_external_usage = True
    serializer_class = SurveyQuestionMatrixChoiceDataSourceSerializer
    filter_class = SurveyQuestionMatrixChoiceDataSourceFilter
    required_filters = ['question']
    allow_empty_filter = True
    max_filtered_result = None

    def get_queryset(self):
        from events.surveyme.models import SurveyQuestionMatrixTitle
        return (
            SurveyQuestionMatrixTitle.objects.all()
        )
