# -*- coding: utf-8 -*-
from events.accounts.utils import GENDER_MALE, GENDER_FEMALE
from events.data_sources.sources.base import (
    BaseDataSourceCategory,
    BaseDataSource,
    SimpleDataSourceSerializer,
)
from events.data_sources.filters import (
    CharListFilter,
    ListFilterSet,
    ListOrCommaSeparatedStringListFilter,
)


class UserEmailListDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'value'
    text_attr = 'value'

    class Meta:
        fields = ('value', 'value')


class UserInfoDataSourceCategory(BaseDataSourceCategory):
    name = 'user_info'
    title = 'Персональные данные'


class UserEmailListDataSourceFilter(ListFilterSet):
    id = ListOrCommaSeparatedStringListFilter(field_name=UserEmailListDataSourceSerializer.id_attr)
    text = CharListFilter(field_name=UserEmailListDataSourceSerializer.text_attr)


class UserEmailListDataSource(BaseDataSource):
    name = 'user_email_list'
    title = 'Email адреса'
    desc = 'Список email адресов пользователя'
    category = UserInfoDataSourceCategory()
    allow_external_usage = True
    is_need_auth = True
    serializer_class = UserEmailListDataSourceSerializer
    filter_class = UserEmailListDataSourceFilter
    allow_empty_filter = True
    max_filtered_result = None

    def get_queryset(self):
        yauser = getattr(self.request, 'yauser', None)
        if yauser:
            if yauser.default_email:  # FORMS-360
                return [{'value': yauser.default_email}]
            else:
                return [
                    {'value': email.get('address')} for email in yauser.emails
                    if email.get('validated') and email.get('address') and email.get('default')
                ]
        if self.user is not None and self.user.uid:
            return [{'value': self.user.email}]
        return []


class GenderDataSourceSerializer(SimpleDataSourceSerializer):
    id_attr = 'code'
    text_attr = 'title'

    class Meta:
        fields = ('code', 'title')


class GenderDataSource(BaseDataSource):
    name = 'gender'
    title = u'Пол'
    desc = u'Пол'
    category = UserInfoDataSourceCategory()
    allow_external_usage = True
    serializer_class = GenderDataSourceSerializer
    allow_empty_filter = True
    max_filtered_result = None

    def get_queryset(self):
        from events.common_app.utils import get_lang_with_fallback
        choices = {
            'en': [(GENDER_MALE, 'Male'), (GENDER_FEMALE, 'Female')],
            'ru': [(GENDER_MALE, 'Мужской'), (GENDER_FEMALE, 'Женский')],
        }
        lang, fallback_lang = get_lang_with_fallback()
        if lang in choices:
            choices = choices[lang]
        elif fallback_lang in choices:
            choices = choices[fallback_lang]
        else:
            choices = choices['ru']
        return [
            {
                'code': code,
                'title': title,
            }
            for code, title in choices
        ]

    def get_filtered_queryset(self, filter_data=None):
        response = super(GenderDataSource, self).get_filtered_queryset(filter_data)
        if filter_data and 'text' in filter_data:
            response = [i for i in response
                        if i[GenderDataSourceSerializer.text_attr] == filter_data['text']]
        return response
