# -*- coding: utf-8 -*-
from django.contrib.contenttypes.models import ContentType
from rest_framework import viewsets
from rest_framework.decorators import action

from events.accounts.models import User
from events.accounts.utils import get_user_uid_by_login
from events.common_app.utils import generate_code
from events.followme.api_admin.v2.serializers import (
    ContentFollowerSerializer,
    ContentFollowerKeepOnlySerializer,
)
from events.followme.models import ContentFollower
from events.rest_framework_contrib.mixins import InternalGenericApiViewV2MixinWithPermissions


class ContentFollowerViewSet(InternalGenericApiViewV2MixinWithPermissions,
                                  viewsets.ModelViewSet):
    serializer_class = ContentFollowerSerializer
    queryset = ContentFollower.objects.all()


class ContentFollowerMixin:
    def _followers(self, obj):
        page = self.paginate_queryset(
            ContentFollower.objects.filter(
                content_type=ContentType.objects.get_for_model(obj),
                object_id=obj.pk,
            )
        )
        serializer = ContentFollowerSerializer(page, many=True)
        return self.get_paginated_response(serializer.data)

    @action(methods=['get'], url_path='followers', detail=True)
    def followers(self, request, *args, **kwargs):
        return self._followers(self.get_object())

    @action(methods=['post'], url_path='followers/keep-only', detail=True)
    def followers_keep_only(self, request, *args, **kwargs):
        serializer = ContentFollowerKeepOnlySerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        followers = serializer.validated_data['followers']
        auth_type = serializer.validated_data['auth_type']

        obj = self.get_object()
        ct = ContentType.objects.get_for_model(obj)
        queryset = ContentFollower.objects.filter(content_type=ct, object_id=obj.pk)
        queryset.delete()

        for follower in followers:
            follower_type = follower.get('type')
            if follower_type == 'user':
                login = follower.get('login')
                uid = follower.get('uid')
                cloud_uid = follower.get('cloud_uid')
                if login or uid or cloud_uid:
                    self._create_user_follower(login, uid, cloud_uid, auth_type, ct, obj)
            elif follower_type == 'mail_list':
                email = follower.get('email')
                if email:
                    self._create_mail_list_follower(email, ct, obj)

        return self._followers(obj)

    def _create_user_follower(self, login, uid, cloud_uid, auth_type, content_type, obj):
        user = None
        if not uid and login:
            uid = get_user_uid_by_login(login, auth_type)
        if uid or cloud_uid:
            try:
                user = User.objects.get_or_create_user(uid, cloud_uid, None)
            except ValueError:
                pass
        if user:
            follower, created = ContentFollower.objects.get_or_create(
                user_id=user.pk,
                object_id=obj.pk,
                content_type=content_type,
                type='user',
            )
            if created:
                follower.secret_code = generate_code()
                follower.save()

    def _create_mail_list_follower(self, email, content_type, obj):
        ContentFollower.objects.get_or_create(
            email=email,
            object_id=obj.pk,
            content_type=content_type,
            type='mail_list'
        )
