# -*- coding: utf-8 -*-
from django.db import models
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import GenericForeignKey
from django.core.exceptions import ValidationError

from events.common_app.fields import IdentifierField
from events.common_app.mixins import TimeStampMixin


FOLLOWER_TYPE_CHOICES = (
    ('user', 'Пользователь'),
    ('mail_list', 'Рассылка'),
)


class ProfileFollowContent(TimeStampMixin, models.Model):
    user = models.ForeignKey('accounts.User', null=True, on_delete=models.CASCADE)

    content_type = models.ForeignKey(ContentType, blank=True, null=True, on_delete=models.DO_NOTHING)
    object_id = IdentifierField(blank=True, null=True)
    content_object = GenericForeignKey()

    class Meta:
        unique_together = (
            ('user', 'content_type', 'object_id'),
        )

    @property
    def survey(self):
        if self.content_type.model == 'survey':
            return self.content_object


class ContentFollower(TimeStampMixin, models.Model):
    user = models.ForeignKey('accounts.User', blank=True, null=True, on_delete=models.CASCADE)
    type = models.CharField(
        'Тип',
        choices=FOLLOWER_TYPE_CHOICES,
        default=FOLLOWER_TYPE_CHOICES[0][0],
        max_length=10,
        blank=False,
        null=False
    )
    email = models.EmailField('Email', blank=True, null=True)

    content_type = models.ForeignKey(ContentType, blank=True, null=True, on_delete=models.DO_NOTHING)
    object_id = IdentifierField(blank=True, null=True)
    content_object = GenericForeignKey()

    secret_code = models.CharField(max_length=255, blank=True, null=True, db_index=True)

    class Meta:
        unique_together = (
            ('user', 'content_type', 'object_id'),
            ('email', 'content_type', 'object_id'),
        )

    @property
    def survey(self):
        if self.content_type.model == 'survey':
            return self.content_object

    def clean_fields(self, *args, **kwargs):
        self._validate_profile_and_email_fields()
        super(ContentFollower, self).clean_fields(*args, **kwargs)

    def _validate_profile_and_email_fields(self):
        if self.type == 'user' and not self.user_id:
            raise ValidationError({'user': 'Это поле обязательно'})  # todo: localize
        if self.type == 'mail_list' and (not self.email or not self.email.strip()):
            raise ValidationError({'email': 'Это поле обязательно'})  # todo: localize

    def get_email(self):
        if self.type == 'user':
            return self.user.get_staff_email()
        else:
            return self.email

    def save(self, *args, **kwargs):
        self._validate_profile_and_email_fields()
        return super(ContentFollower, self).save(*args, **kwargs)


class FollowersMixin(object):
    @property
    def followers(self):
        return ContentFollower.objects.filter(
            content_type=ContentType.objects.get_for_model(self),
            object_id=self.id
        )
