# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from events.geobase_contrib.client import GeobaseClient
from events.geobase_contrib.models import City, Country

logger = logging.getLogger(__name__)


class GeobaseSync:
    def insert_one(self, model, geobase_id, **kwargs):
        model.objects.create(geobase_id=geobase_id, **kwargs)

    def update_one(self, model, geobase_id, **kwargs):
        model.objects.filter(geobase_id=geobase_id).update(**kwargs)

    def sync_countries(self, countries):
        key_to_md5sum = {
            geobase_id: md5sum
            for geobase_id, md5sum in (
                Country.objects.using(settings.DATABASE_ROLOCAL)
                .values_list('geobase_id', 'md5sum')
            )
        }
        total_count, insert_count, update_count = 0, 0, 0
        for item in countries:
            geobase_id = int(item['id'])
            name = item['name']
            full_name = item['full_name']
            translations = item['translations']
            md5sum = Country.calc_md5sum(name, full_name, translations)

            if geobase_id not in key_to_md5sum:
                self.insert_one(
                    Country,
                    geobase_id,
                    name=name,
                    full_name=full_name,
                    translations=translations,
                    md5sum=md5sum,
                )
                insert_count += 1
            elif key_to_md5sum[geobase_id] != md5sum:
                self.update_one(
                    Country,
                    geobase_id,
                    name=name,
                    full_name=full_name,
                    translations=translations,
                    md5sum=md5sum,
                )
                update_count += 1
            total_count += 1
        logger.info('Sync countries: found %s, inserted new %s, updated old %s', total_count, insert_count, update_count)
        return insert_count, update_count

    def sync_cities(self, cities):
        countries = {
            geobase_id: pk
            for geobase_id, pk in Country.objects.values_list('geobase_id', 'pk')
        }
        key_to_md5sum = {
            geobase_id: md5sum
            for geobase_id, md5sum in (
                City.objects.using(settings.DATABASE_ROLOCAL)
                .values_list('geobase_id', 'md5sum')
            )
        }
        total_count, insert_count, update_count = 0, 0, 0
        for item in cities:
            geobase_id = int(item['id'])
            country_id = countries.get(int(item['country']['id']))
            name = item['name']
            full_name = item['full_name']
            population = item['population']
            translations = item['translations']
            md5sum = City.calc_md5sum(name, full_name, population, translations)

            if geobase_id not in key_to_md5sum:
                self.insert_one(
                    City,
                    geobase_id,
                    country_id=country_id,
                    name=name,
                    full_name=full_name,
                    profiles_count=population,
                    translations=translations,
                    md5sum=md5sum,
                )
                insert_count += 1
            elif key_to_md5sum[geobase_id] != md5sum:
                self.update_one(
                    City,
                    geobase_id,
                    country_id=country_id,
                    name=name,
                    full_name=full_name,
                    profiles_count=population,
                    translations=translations,
                    md5sum=md5sum,
                )
                update_count += 1
            total_count += 1
        logger.info('Sync cities: found %s, inserted new %s, updated old %s', total_count, insert_count, update_count)
        return insert_count, update_count

    def sync_with_source(self):
        client = GeobaseClient()
        geobase = client.get_geobase_data()

        cities = list(client.get_cities(geobase))
        countries = client.get_countries(cities)

        self.sync_countries(countries)
        self.sync_cities(cities)


def sync_with_geobase():
    sync = GeobaseSync()
    sync.sync_with_source()
