# -*- coding: utf-8 -*-
import logging

from django.utils.encoding import force_str

from rest_framework.exceptions import PermissionDenied
from rest_framework.response import Response
from rest_framework.permissions import SAFE_METHODS
from rest_framework.decorators import action


from guardian.shortcuts import get_users_with_perms

from events.guardian_contrib.api_admin.v2.serializers import UserWithPermissionsSerializer
from events.guardian_contrib.utils import has_any_perm


class UsersWithPermissionsMixin(object):
    logger = logging.getLogger(__name__)

    @action(url_path='users-with-permissions', detail=True)
    def get_users_with_permissions(self, request, *args, **kwargs):
        self._check_permissions(request)
        return Response(data=self._get_users_with_permissions())

    def _check_permissions(self, request):
        write_perms = ['auth.full_permission', 'auth.change_permission']
        read_perms = write_perms + ['auth.view_permission']
        if request.method in SAFE_METHODS and not has_any_perm(request.user, read_perms):
            raise PermissionDenied
        elif request.method not in SAFE_METHODS and not has_any_perm(request.user, write_perms):
            self.logger.warning('Permission denied for user %s', force_str(self._get_user_or_group()))
            raise PermissionDenied

    def _get_users_with_permissions(self):
        users_with_permissions = get_users_with_perms(
            self.get_object(),
            attach_perms=True,
            with_superusers=True,
            with_group_users=True
        )
        result = []
        for user, permissions in users_with_permissions.items():
            user_data = UserWithPermissionsSerializer(user).data
            user_data['permissions'] = permissions
            result.append(user_data)
        return result
