# -*- coding: utf-8 -*-
import json

from django.conf import settings

from events.common_app.utils import requests_session
from events.yauth_contrib.auth import OAuth


class IdmClient:
    def __init__(self, token=None):
        self.auth = OAuth(token or settings.IDM_TOKEN)

    def _get_request(self, path, params=None):
        url = f'{settings.IDM_URL}{settings.IDM_PREFIX}{path}'
        while True:
            response = requests_session.get(
                url, params=params, auth=self.auth,
                timeout=settings.DEFAULT_TIMEOUT,
            )
            response.raise_for_status()
            data = response.json()
            for obj in data.get('objects') or []:
                yield obj
            next_url = data.get('meta', {}).get('next')
            if not next_url:
                break
            url = f'{settings.IDM_URL}{next_url}'

    def _make_request(self, method, path, data=None):
        url = f'{settings.IDM_URL}{settings.IDM_PREFIX}{path}'
        response = requests_session.request(
            method, url, json=data,
            auth=self.auth, timeout=settings.DEFAULT_TIMEOUT,
        )
        response.raise_for_status()
        if not response.content:
            return ''
        return response.json()

    def get_approverequests(self, approver, status=None):
        params = {
            'approver': approver,
            'status': status or 'pending',
        }
        yield from self._get_request('/approverequests/', params)

    def approve_requests(self, req_id, approve=True):
        data = {
            'approve': approve,
        }
        return self._make_request('post', f'/approverequests/{req_id}/', data)

    def get_one_role(self, user=None, group=None, **params):
        data = dict(params)
        if user:
            data['user'] = user
            data['label'] = f'Request role for user {user}'
        elif group:
            data['group'] = group
            data['label'] = f'Request role for group {group}'
        return data

    def request_roles(self, users=None, groups=None, **params):
        if not users and not groups:
            return
        roles = []
        if users:
            roles += [
                self.get_one_role(user=user, **params)
                for user in users
            ]
        if groups:
            roles += [
                self.get_one_role(group=group, **params)
                for group in groups
            ]
        data = {'requests': roles}
        return self._make_request('post', '/batchrolerequest/', data)

    def reject_roles(self, users=None, groups=None, **params):
        if not users and not groups:
            return
        fields_data = params.pop('fields_data', None)
        data = dict(params)
        if users:
            data['user'] = ','.join(users)
        if groups:
            data['group'] = ','.join(map(str, groups))
        if fields_data:
            data['fields_data'] = json.dumps(fields_data)
        return self._make_request('delete', '/roles/', data)
