# -*- coding: utf-8 -*-
from rest_framework.authentication import BaseAuthentication


class YauthAuthentication(BaseAuthentication):
    def authenticate(self, request):
        """
        Returns a `User` if the request session currently has a logged in user.
        Otherwise returns `None`.
        """

        # Get the underlying HttpRequest object
        http_request = request._request

        yauser = http_request.yauser

        if not yauser.is_authenticated():
            if yauser.authenticated_by:
                if yauser.authenticated_by.mechanism_name in ('cookie', 'http_header'):
                    http_request._authentication_failed_with = 'sessionid'
                elif yauser.authenticated_by.mechanism_name == 'oauth':
                    http_request._authentication_failed_with = 'oauth'
                else:
                    http_request._authentication_failed_with = 'dragonfly'
            else:
                http_request._authentication_failed_with = 'dragonfly'

        return http_request.user, None

    def authenticate_header(self, request):
        """
        Return a string to be used as the value of the `WWW-Authenticate`
        header in a `401 Unauthenticated` response, or `None` if the
        authentication scheme should return `403 Permission Denied` responses.

        http://wiki.yandex-team.ru/oauth/newservice#kakpodderzhatoauthvsvojomservise
        """
        if hasattr(request, '_authentication_failed_with'):
            authentication_failed_with = request._authentication_failed_with
        else:
            authentication_failed_with = None
        if authentication_failed_with == 'sessionid':
            scheme = 'SessionId'
            error = 'invalid_session_id'
        elif authentication_failed_with == 'oauth':
            scheme = 'OAuth'
            error = 'expired_token'
        else:
            scheme = 'OAuth'
            error = None
        www_authenticate = "{scheme} realm='{service}'".format(
            scheme=scheme,
            service='Technologies'
        )
        if error:
            www_authenticate += ", error='{error}'".format(error=error)
        return www_authenticate
