# -*- coding: utf-8 -*-
import json

from rest_framework import serializers
from rest_framework.fields import Field, ImageField as BaseImageField, ReadOnlyField


class AvatarImageField(BaseImageField):
    def __init__(self, sizes=None, aliases=None, *args, **kwargs):
        self.sizes = sizes or []
        self.aliases = aliases or []
        self.aliases.append('small')
        self.aliases.append('big')
        self.aliases.append('orig')
        super(AvatarImageField, self).__init__(*args, **kwargs)

    def to_representation(self, value):
        if value:
            dict_value = {}
            for size in self.sizes:
                key = '%sx%s' % size
                dict_value[key] = value.get_thumb_url(width=size[0], height=size[1])

            for alias in self.aliases:
                dict_value[alias] = value.get_thumb_url(alias=alias)
        else:
            dict_value = None

        return dict_value


class ImageField(BaseImageField):
    def __init__(self, sizes=None, format_func=None, *args, **kwargs):
        self.sizes = sizes or []
        self.format_func = format_func
        super(ImageField, self).__init__(*args, **kwargs)

    def default_format_func(self, value):
        result = {}
        for size in self.get_sizes():
            key = '%sx%s' % size
            result[key] = value.get_thumb_url(width=size[0], height=size[1])

        return result

    def format_for_image_view(self, value):
        return [
            {'width': width, 'height': height, 'url': value.get_thumb_url(width=width, height=height), }
            for width, height in self.get_sizes()
        ]

    def get_sizes(self):
        return (
            (width or '', height or '')
            for width, height in self.sizes
        )

    def to_representation(self, value):
        result = None
        if value:
            result = getattr(self, self.format_func or 'default_format_func')(value)

        return result


class SplittedDescriptionField(ReadOnlyField):
    def to_representation(self, value):
        if value:
            value = value.split('.')
            splitted_value = [
                '{0}.'.format(value[0]),
                '.'.join(value[1:]).strip()
            ]
        else:
            splitted_value = []
        return splitted_value


class JSONField(Field):
    def to_representation(self, value):
        """
        Reverts a simple representation back to the field's value.
        """
        return json.loads(value)

    def to_internal_value(self, value):
        return json.dumps(value)


class FilterHeadersJSONField(serializers.JSONField):
    HEADERS_TO_FILTER = {
        'authorization',
        'x-ya-service-ticket',
        'x-ya-user-ticket',
        'oauth',
    }

    def to_representation(self, value):
        result = super(FilterHeadersJSONField, self).to_representation(value)
        self.filter_headers(result)
        return result

    def filter_headers(self, result):
        headers = result.get('headers')
        if isinstance(headers, dict):
            result['headers'] = {
                header: value if not self.exclude_header(header) else '****'
                for header, value in headers.items()
            }

    def exclude_header(self, header):
        return header.lower() in self.HEADERS_TO_FILTER
