# -*- coding: utf-8 -*-
from django.conf import settings
from django.http import Http404
from rest_framework.permissions import BasePermission


class IsAuthenticated(BasePermission):
    def has_permission(self, request, view):
        return request.yauser.is_authenticated()

    def has_object_permission(self, request, view, obj):
        return self.has_permission(request, view)


class IsSuperUser(BasePermission):
    def has_permission(self, request, view):
        if request.yauser.is_authenticated():
            return request.user.is_superuser
        return False

    def has_object_permission(self, request, view, obj):
        return self.has_permission(request, view)


class IsSupport(BasePermission):
    def has_permission(self, request, view):
        if request.yauser.is_authenticated():
            return request.user.is_staff
        return False

    def has_object_permission(self, request, view, obj):
        return self.has_permission(request, view)


class HasPermission(BasePermission):
    action = None

    def _check_org(self, obj, orgs):
        if settings.IS_BUSINESS_SITE:
            if hasattr(obj, 'org') and obj.org:
                if obj.org.dir_id not in (orgs or []):
                    raise Http404

    def _check_obj(self, user, obj, orgs):
        self._check_org(obj, orgs)
        return user.has_perm(f'{self.action}_{obj._meta.model_name}', obj)

    def has_object_permission(self, request, view, obj):
        from events.surveyme.models import Survey
        if request.user.is_superuser:
            return True

        if hasattr(obj, 'survey') and obj.survey:
            obj = obj.survey

        if self._check_obj(request.user, obj, request.orgs):
            return True

        if isinstance(obj, Survey) and obj.group:
            return self._check_obj(request.user, obj.group, request.orgs)

        return False


class HasChangePermission(HasPermission):
    action = 'change'


class HasViewFilePermission(HasPermission):
    action = 'viewfile'


class IsAnswerAuthor(BasePermission):
    def _get_answer_author(self, answer_id):
        from rest_framework.exceptions import ValidationError
        from events.surveyme.models import ProfileSurveyAnswer
        try:
            return ProfileSurveyAnswer.objects.values_list('user_id', flat=True).get(pk=answer_id)
        except ProfileSurveyAnswer.DoesNotExist:
            raise ValidationError('Find no answer with such id')

    def has_permission(self, request, view):
        if request.user.is_superuser:
            return True

        return request.user.pk == self._get_answer_author(view.answer_id)


class AuthorizedFrontendOnly(BasePermission):
    def has_permission(self, request, view):
        return request.frontend


class IsAuthenticatedForBusiness(BasePermission):
    def has_object_permission(self, request, view, obj):
        if settings.IS_BUSINESS_SITE:
            if not getattr(obj, 'is_public', False):
                return request.yauser.is_authenticated()
        return True


def allowed_for_services(*tvm2_client_ids):
    clients = set(tvm2_client_ids)
    class _Permission(BasePermission):
        def has_permission(self, request, view):
            service_ticket = getattr(request.yauser, 'service_ticket', None)
            if service_ticket:
                return str(service_ticket.src) in clients
            return False

        def has_object_permission(self, request, view, obj):
            return self.has_permission(request, view)
    return _Permission


TakeoutClientOnly = allowed_for_services(settings.TVM2_CLIENT_ID, settings.TAKEOUT_TVM2_CLIENT)
Takeout2ClientOnly = allowed_for_services(settings.TVM2_CLIENT_ID, settings.TAKEOUT2_TVM2_CLIENT)
AntispamClientsOnly = allowed_for_services(settings.TVM2_CLIENT_ID, *settings.ANTISPAM_TVM2_CLIENTS)
IdmClientOnly = allowed_for_services(settings.TVM2_CLIENT_ID, settings.IDM_TVM2_CLIENT)


def ANY(*permission_classes):
    class _Permission(BasePermission):
        def has_permission(self, request, view):
            for permission in self.get_permissions():
                if permission.has_permission(request, view):
                    return True
            return False

        def has_object_permission(self, request, view, obj):
            for permission in self.get_permissions():
                if permission.has_object_permission(request, view, obj):
                    return True
            return False

        def get_permissions(self):
            return (permission() for permission in permission_classes)
    return _Permission


class HasFileReadPermission(BasePermission):
    def has_object_permission(self, request, view, obj):
        if request.user.is_anonymous:
            # анонимный пользователь не имеет доступа к файлам
            return False

        if obj.user_id == request.user.pk or request.user.is_superuser:
            # если тот же пользователь, что и подгрузивший файл - разрешаем доступ
            return True

        if not obj.survey:
            return False

        change_permission = HasChangePermission()
        # проверяем доступ пользователя на редактирование формы
        if change_permission.has_object_permission(request, view, obj.survey):
            return True

        viewfile_permission = HasViewFilePermission()
        # проверяем доступ пользователя на чтение файлов
        return viewfile_permission.has_object_permission(request, view, obj.survey)


class HasFilePermission(HasFileReadPermission):
    def has_object_permission(self, request, view, obj):
        if request.method != 'GET':
            return IsAuthenticated().has_permission(request, view)
        return super().has_object_permission(request, view, obj)
