# -*- coding: utf-8 -*-
from urllib.parse import urljoin

from django.conf import settings
from ids.registry import registry

from events.common_app.clients import BaseClient
from events.common_app.tvm2_client import get_service_ticket
from events.yauth_contrib.auth import TvmAuth


def get_staff_repository(resource_type):
    return registry.get_repository(
        'staff',
        resource_type,
        user_agent='forms',
        service_ticket=get_service_ticket(settings.STAFF_TVM2_CLIENT),
        retries=3,
        timeout=settings.DEFAULT_TIMEOUT,
    )


class StaffBaseClient(BaseClient):
    results_key = 'result'
    resource_name = None
    fields = None
    limit = 100

    def get_params(self):
        return {
            '_fields': self.fields,
            '_limit': self.limit,
        }

    def get_auth(self):
        return TvmAuth(settings.STAFF_TVM2_CLIENT)

    def get_url(self):
        return urljoin(settings.STAFF_API_URL, self.resource_name)

    def get_next_url(self, data):
        return data['links'].get('next')


class StaffPersonClient(StaffBaseClient):
    resource_name = 'persons'
    fields = 'id,login,name.first,name.last,official.is_dismissed,uid,department_group.id,is_deleted'

    def process_one_item(self, item):
        yield {
            'staff_id': item['id'],
            'login': item['login'],
            'yandex_uid': item['uid'],
            'first_name': item['name']['first'].get('ru'),
            'last_name': item['name']['last'].get('ru'),
            'group_id': item['department_group']['id'],
            'is_deleted': item['is_deleted'],
            'is_dismissed': item['official']['is_dismissed'],
            'translations': {
                'first_name': item['name']['first'],
                'last_name': item['name']['last'],
            },
        }


class StaffGroupClient(StaffBaseClient):
    resource_name = 'groups'
    fields = 'id,name,url,type,is_deleted,role_scope'

    def process_one_item(self, item):
        yield {
            'staff_id': item['id'],
            'url': item['url'],
            'name': item['name'],
            'type': item['type'],
            'is_deleted': item['is_deleted'],
            'role_scope': item['role_scope'],
        }


class StaffOrganizationClient(StaffBaseClient):
    resource_name = 'organizations'
    fields = 'id,name,is_deleted'

    def process_one_item(self, item):
        yield {
            'staff_id': item['id'],
            'name': item['name'],
            'is_deleted': item['is_deleted'],
        }


class StaffOfficeClient(StaffBaseClient):
    resource_name = 'office'
    fields = 'id,name,address,is_deleted'

    def process_one_item(self, item):
        yield {
            'staff_id': item['id'],
            'name': item['name'].get('ru'),
            'address': item['address'].get('ru'),
            'is_deleted': item['is_deleted'],
            'translations': {
                'name': item['name'],
                'address': item['address'],
            },
        }
