# -*- coding: utf-8 -*-
import logging
import operator

from django.db import models, IntegrityError
from django.db.models import Q
from django.db.models.query import QuerySet
from django.db.models.functions import Length
from functools import reduce

from events.common_app.managers import ProxyToQuerySetMixin
from events.staff.client import (
    StaffGroupClient,
    StaffPersonClient,
    StaffOrganizationClient,
    StaffOfficeClient,
)
from events.translation.managers import TranslationQuerySetMixin

logger = logging.getLogger(__name__)


class StaffBaseQuerySet(models.QuerySet):
    suggest_field = 'name'

    def filter_by_fit_for_search_text(self, search_text):
        words = search_text.split()
        field_name = '{field}__icontains'.format(field=self.suggest_field)
        q_list = [
            Q(**{field_name: word})
            for word in words
        ]
        return (
            self.filter(reduce(operator.and_, q_list))
            .order_by(Length(self.suggest_field), self.suggest_field)
        )


class StaffBaseManager(ProxyToQuerySetMixin, models.Manager):
    client_class = None
    queryset_class = None

    def get_queryset(self):
        return self.queryset_class(self.model, using=self._db)

    def sync_with_source(self):
        key_to_md5sum = dict(
            pair
            for pair in self.model.objects.values_list('staff_id', 'md5sum')
        )
        sync_client = self.client_class()
        for item in sync_client.make_request():
            self.sync_one_item(item, key_to_md5sum)

    def insert_one(self, staff_id, **kwargs):
        try:
            self.model.objects.create(staff_id=staff_id, **kwargs)
        except IntegrityError:
            logger.exception()

    def update_one(self, staff_id, **kwargs):
        self.model.objects.filter(staff_id=staff_id).update(**kwargs)


class StaffGroupQuerySet(StaffBaseQuerySet):
    pass


class StaffGroupManager(StaffBaseManager):
    client_class = StaffGroupClient
    queryset_class = StaffGroupQuerySet

    def sync_one_item(self, item, key_to_md5sum):
        staff_id = item['staff_id']
        name = item['name']
        url = item['url']
        type = item['type']
        is_deleted = item['is_deleted']
        role_scope = item['role_scope']
        md5sum = self.model.calc_md5sum(name, url, type, is_deleted, role_scope)

        if staff_id not in key_to_md5sum:
            self.insert_one(
                staff_id,
                name=name,
                url=url,
                type=type,
                is_deleted=is_deleted,
                role_scope=role_scope,
                md5sum=md5sum,
            )
        elif key_to_md5sum[staff_id] != md5sum:
            self.update_one(
                staff_id,
                name=name,
                url=url,
                type=type,
                is_deleted=is_deleted,
                role_scope=role_scope,
                md5sum=md5sum,
            )


class StaffPersonQuerySet(TranslationQuerySetMixin, QuerySet):
    def _filter_by_login_search(self, search_text):
        words = search_text.split()
        return self.filter(
            **{
                'login__startswith': word
                for word in words
            }
        ).order_by(Length('login'))

    def filter_by_fit_for_search_text(self, search_text):
        return self.filter_translated_fields(
            search_text,
            'first_name',
            'last_name'
        ) | self._filter_by_login_search(search_text)


class StaffPersonManager(StaffBaseManager):
    client_class = StaffPersonClient
    queryset_class = StaffPersonQuerySet

    def sync_one_item(self, item, key_to_md5sum):
        staff_id = item['staff_id']
        login = item['login']
        yandex_uid = item['yandex_uid']
        first_name = item['first_name']
        last_name = item['last_name']
        translations = item['translations']
        is_deleted = item['is_deleted']
        is_dismissed = item['is_dismissed']
        group_id = item['group_id']
        md5sum = self.model.calc_md5sum(login, yandex_uid, first_name, last_name,
                                        translations, is_deleted, is_dismissed, group_id)

        if staff_id not in key_to_md5sum:
            self.insert_one(
                staff_id,
                login=login,
                yandex_uid=yandex_uid,
                first_name=first_name,
                last_name=last_name,
                translations=translations,
                is_deleted=is_deleted,
                is_dismissed=is_dismissed,
                group_id=group_id,
                md5sum=md5sum,
            )
        elif key_to_md5sum[staff_id] != md5sum:
            self.update_one(
                staff_id,
                login=login,
                yandex_uid=yandex_uid,
                first_name=first_name,
                last_name=last_name,
                translations=translations,
                is_deleted=is_deleted,
                is_dismissed=is_dismissed,
                group_id=group_id,
                md5sum=md5sum,
            )


class StaffOrganizationQuerySet(StaffBaseQuerySet):
    pass


class StaffOrganizationManager(StaffBaseManager):
    client_class = StaffOrganizationClient
    queryset_class = StaffOrganizationQuerySet

    def sync_one_item(self, item, key_to_md5sum):
        staff_id = item['staff_id']
        name = item['name']
        is_deleted = item['is_deleted']
        md5sum = self.model.calc_md5sum(name, is_deleted)

        if staff_id not in key_to_md5sum:
            self.insert_one(
                staff_id,
                name=name,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )
        elif key_to_md5sum[staff_id] != md5sum:
            self.update_one(
                staff_id,
                name=name,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )


class StaffOfficeQuerySet(TranslationQuerySetMixin, QuerySet):
    def filter_by_fit_for_search_text(self, search_text):
        return self.filter_translated_fields(search_text, 'name')


class StaffOfficeManager(StaffBaseManager):
    client_class = StaffOfficeClient
    queryset_class = StaffOfficeQuerySet

    def sync_one_item(self, item, key_to_md5sum):
        staff_id = item['staff_id']
        name = item['name']
        address = item['address']
        translations = item['translations']
        is_deleted = item['is_deleted']
        md5sum = self.model.calc_md5sum(name, address, translations, is_deleted)

        if staff_id not in key_to_md5sum:
            self.insert_one(
                staff_id,
                name=name,
                address=address,
                translations=translations,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )
        elif key_to_md5sum[staff_id] != md5sum:
            self.update_one(
                staff_id,
                name=name,
                address=address,
                translations=translations,
                is_deleted=is_deleted,
                md5sum=md5sum,
            )
