# -*- coding: utf-8 -*-
import json
from django.db import models

from events.common_app.models import Md5sumMixin
from events.translation.models import TranslationModel
from events.staff.managers import (
    StaffGroupManager,
    StaffPersonManager,
    StaffOrganizationManager,
    StaffOfficeManager,
)


STAFF_GROUP_CHOICES = (
    ('department', 'department'),
    ('service', 'service'),
    ('servicerole', 'servicerole'),
    ('wiki', 'wiki'),
)


class StaffGroup(Md5sumMixin, models.Model):
    staff_id = models.PositiveIntegerField('ID на стаффе', primary_key=True)
    name = models.CharField(max_length=255)
    url = models.CharField(max_length=255)
    type = models.CharField(max_length=255, choices=STAFF_GROUP_CHOICES)
    is_deleted = models.BooleanField(default=False, db_index=True)
    role_scope = models.CharField(max_length=255, blank=True, null=True)

    objects = StaffGroupManager()

    class Meta:
        index_together = (
            ('name', 'url'),
        )

    @classmethod
    def calc_md5sum(cls, name, url, type, is_deleted, role_scope):
        return cls._calc_md5sum(
            name,
            url,
            type,
            is_deleted,
            role_scope,
        )

    def md5(self):
        return self.calc_md5sum(
            self.name,
            self.url,
            self.type,
            self.is_deleted,
            self.role_scope,
        )

    def get_info_url(self):
        if self.type == 'department':
            return f'https://staff.yandex-team.ru/departments/{self.url}/'
        elif self.type == 'service':
            service = self.url[4:]
            return f'https://abc.yandex-team.ru/services/{service}'
        elif self.type == 'servicerole':
            service = self.url[4:-len(str(self.role_scope)) - 1]
            return f'https://abc.yandex-team.ru/services/{service}?scope={self.role_scope}'
        return f'https://staff.yandex-team.ru/groups/{self.url}'

    def __str__(self):
        return 'Группа "%s"' % self.name


class StaffPerson(Md5sumMixin, TranslationModel):
    FIELDS_FOR_TRANSLATION = ('first_name', 'last_name')

    staff_id = models.PositiveIntegerField('ID на стаффе', primary_key=True)
    login = models.CharField(max_length=255, unique=True, db_index=True)
    yandex_uid = models.CharField(max_length=255, unique=True)
    first_name = models.CharField(max_length=255)
    last_name = models.CharField(max_length=255)
    is_deleted = models.BooleanField(default=False, db_index=True)
    is_dismissed = models.BooleanField(default=False)
    group_id = models.PositiveIntegerField(null=True, db_index=True)

    objects = StaffPersonManager()

    class Meta:
        index_together = (
            ('login', 'first_name', 'last_name'),
        )

    def __str__(self):
        return '%s %s (%s)' % (self.first_name, self.last_name, self.login)

    @classmethod
    def calc_md5sum(cls, login, yandex_uid, first_name, last_name, translations, is_deleted, is_dismissed, group_id):
        return cls._calc_md5sum(
            login,
            yandex_uid,
            first_name,
            last_name,
            json.dumps(translations, sort_keys=True, ensure_ascii=False),
            is_deleted,
            is_dismissed,
            group_id,
        )

    def md5(self):
        return self.calc_md5sum(
            self.login,
            self.yandex_uid,
            self.first_name,
            self.last_name,
            self.translations,
            self.is_deleted,
            self.is_dismissed,
            self.group_id,
        )

    def get_first_name(self):
        return self.get_translated_field('first_name')

    def get_last_name(self):
        return self.get_translated_field('last_name')


class StaffOrganization(Md5sumMixin, models.Model):
    staff_id = models.PositiveIntegerField('ID на стаффе', primary_key=True)
    name = models.CharField(max_length=255, db_index=True)
    is_deleted = models.BooleanField(default=False, db_index=True)

    objects = StaffOrganizationManager()

    @classmethod
    def calc_md5sum(cls, name, is_deleted):
        return cls._calc_md5sum(
            name,
            is_deleted,
        )

    def md5(self):
        return self.calc_md5sum(
            self.name,
            self.is_deleted,
        )

    def __str__(self):
        return self.name


class StaffOffice(Md5sumMixin, TranslationModel):
    FIELDS_FOR_TRANSLATION = ('name', 'address')

    staff_id = models.PositiveIntegerField('ID на стаффе', primary_key=True)
    name = models.CharField(max_length=255, db_index=True, null=True)
    address = models.CharField(max_length=255, null=True)
    is_deleted = models.BooleanField(default=False, db_index=True)

    objects = StaffOfficeManager()

    def __str__(self):
        return self.name

    @classmethod
    def calc_md5sum(cls, name, address, translations, is_deleted):
        return cls._calc_md5sum(
            name,
            address,
            json.dumps(translations, sort_keys=True, ensure_ascii=False),
            is_deleted,
        )

    def md5(self):
        return self.calc_md5sum(
            self.name,
            self.address,
            self.translations,
            self.is_deleted,
        )

    def get_name(self):
        return self.get_translated_field('name')

    def get_address(self):
        return self.get_translated_field('address')
