# -*- coding: utf-8 -*-
from django.conf import settings

from events.common_app.utils import requests_session
from events.yauth_contrib.auth import TvmAuth


class StaffClient:
    path = '/'
    limit = 50

    def __init__(self):
        self.auth = TvmAuth(settings.STAFF_TVM2_CLIENT)

    def _is_paginated(self, data):
        return 'links' in data and 'result' in data

    def _fetch_items(self, data):
        if not self._is_paginated(data):
            yield data
        else:
            for it in data.get('result', []):
                yield it

    def _make_response(self, path=None, fields=None, limit=None, params=None):
        path = path or ''
        params = params or {}
        limit = limit or self.limit
        if '_limit' not in params:
            params['_limit'] = limit
        if '_fields' not in params and fields is not None:
            params['_fields'] = fields

        url = f'{settings.STAFF_URL}{settings.STAFF_PREFIX}{self.path}{path}'
        response = requests_session.get(
            url, params=params,
            auth=self.auth, timeout=settings.DEFAULT_TIMEOUT,
            verify=settings.YANDEX_ROOT_CERTIFICATE,
        )
        response.raise_for_status()
        data = response.json()
        yield from self._fetch_items(data)

        while self._is_paginated(data):
            links = data.get('links', {})
            url = links.get('next')
            if not url:
                break

            response = requests_session.get(
                url, auth=self.auth, timeout=settings.DEFAULT_TIMEOUT,
            )
            response.raise_for_status()
            data = response.json()
            yield from self._fetch_items(data)

    def get(self, staff_id, params=None, **kwargs):
        it = iter(self._make_response(path=staff_id, params=params, **kwargs))
        try:
            return next(it)
        except StopIteration:
            pass

    def list(self, params=None, **kwargs):
        yield from self._make_response(params=params, **kwargs)


class StaffPersonClient(StaffClient):
    path = '/persons/'


class StaffGroupClient(StaffClient):
    path = '/groups/'


class StaffGroupMembershipClient(StaffClient):
    path = '/groupmembership/'

    def list(self, group_id, params=None, **kwargs):
        params = params or {}
        params['group.id'] = group_id
        yield from super().list(params=params, **kwargs)
