# -*- coding: utf-8 -*-
from django.conf import settings
from django.core.cache import caches
from django.http import Http404
from rest_framework import viewsets, generics
from rest_framework.decorators import action
from rest_framework.response import Response

from events.common_app.startrek.client import get_startrek_client, StartrekFieldsSuggest
from events.rest_framework_contrib.mixins import InternalGenericApiViewV2Mixin
from events.rest_framework_contrib.pagination import LargePagination
from events.startrek.api_admin.v2.serializers import StartrekFieldSerializer, StartrekQueueSerializer
from events.startrek.models import StartrekField


def get_dir_id(survey_id, orgs):
    from events.surveyme.models import Survey
    dir_id = None
    if survey_id:
        try:
            dir_id = Survey.objects.values_list('org__dir_id', flat=True).get(pk=survey_id)
        except Survey.DoesNotExist:
            return None
    if not dir_id:
        return None
    if dir_id not in orgs:
        return None
    return dir_id


class StartrekFieldViewSet(InternalGenericApiViewV2Mixin, viewsets.ModelViewSet):
    serializer_class = StartrekFieldSerializer
    queryset = StartrekField.objects.filter_optional_fields()
    pagination_class = LargePagination

    @action(url_path='suggest', detail=False)
    def suggest(self, request, *args, **kwargs):
        data = []
        dir_id = None
        search = request.query_params.get('search', '').lower()
        queue_name = request.query_params.get('queue', '').upper()
        if search:
            survey = request.query_params.get('survey')
            if settings.IS_BUSINESS_SITE:
                dir_id = get_dir_id(survey, request.orgs)
                if not dir_id:  # даже не пытаемся искать поля в б2б если не указан dir_id
                    return Response([])

            lang = self.get_language_code(fallback=False)
            fields = StartrekFieldsSuggest(lang=lang, dir_id=dir_id)
            data = [
                self._get_field_info(field)
                for field in fields.suggest(search, queue_name=queue_name)
            ]
        return Response(data)

    def _get_field_info(self, field):
        return {
            'id': field.get('id'),
            'slug': field.get('key') or field.get('id'),
            'name': field.get('name'),
            'type': self._get_field_type(field),
        }

    def _get_field_type(self, field):
        schema = field.get('schema') or {}
        schema_type = schema.get('type')
        schema_items = schema.get('items')
        if schema_type != 'array':
            return schema_type
        return f'{schema_type}/{schema_items}'


class StartrekQueueView(InternalGenericApiViewV2Mixin, generics.RetrieveAPIView):
    def get_queue_data(self, queue_name, dir_id):
        cache = caches['startrek']
        cache_key = f'queues:{dir_id}:{queue_name}'
        queue_data = cache.get(cache_key)
        if queue_data is None:
            queue_data = self._get_queue_data(queue_name, dir_id)
            cache.set(cache_key, queue_data)
        return queue_data

    def _get_queue_data(self, queue_name, dir_id):
        lang = self.get_language_code(fallback=False)
        client = get_startrek_client(dir_id, lang=lang)
        queue = client.get_queue(queue_name)
        if not queue:
            raise Http404
        queue['priorities'] = client.get_priorities() or []
        queue['issuetypes'] = client.get_issuetypes(queue_name) or []
        queue['components'] = []
        return queue

    def retrieve(self, request, queue_name):
        survey = request.query_params.get('survey')
        dir_id = None
        if settings.IS_BUSINESS_SITE:
            dir_id = get_dir_id(survey, request.orgs)
            if not dir_id:  # даже не пытаемся искать очередь в б2б если не указан dir_id
                raise Http404
        queue_data = self.get_queue_data(queue_name, dir_id)
        serializer = StartrekQueueSerializer(queue_data)
        return Response(serializer.data)
